﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Shapes;
using VIZ.ElectricRabbit.Domain;
using VIZ.Framework.Common;
using VIZ.Framework.Core;

namespace VIZ.ElectricRabbit.Module
{
    /// <summary>
    /// 主视图模型
    /// </summary>
    public class MainViewModel : ViewModelBase, IOutputSupport
    {
        public MainViewModel()
        {
            // 初始化命令
            this.InitCommand();

            // 初始化控制器
            this.InitController();

            // 初始化FPS
            this.InitFPS();

            // 初始化消息
            this.InitMessage();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {
            this.SettingCommand = new VCommand(this.Setting);
            this.MinCommand = new VCommand(this.Min);
            this.CloseCommand = new VCommand(this.Close);
            this.LoadedCommand = new VCommand(this.Loaded);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void InitMessage()
        {
            ApplicationDomainEx.MessageManager.Register<AlgorithmDectectMessage>(this, this.OnAlgorithmDectectMessage);
        }

        /// <summary>
        /// 初始化FPS
        /// </summary>
        private void InitFPS()
        {
            this.AlgorithmFPS = new FPSHelper();
            this.AlgorithmFPS.Start();
        }

        /// <summary>
        /// 初始化控制器
        /// </summary>
        private void InitController()
        {
            this.outputController = new OutputController(this);

            this.outputController.Start();
        }

        // ================================================================================
        // Field
        // ================================================================================

        /// <summary>
        /// 输出控制器
        /// </summary>
        private OutputController outputController;

        // ================================================================================
        // Property
        // ================================================================================

        #region IsShowSettingView -- 是否显示设置视图

        private bool isShowSettingView = true;
        /// <summary>
        /// 是否显示设置视图
        /// </summary>
        public bool IsShowSettingView
        {
            get { return isShowSettingView; }
            set { isShowSettingView = value; this.RaisePropertyChanged(nameof(IsShowSettingView)); }
        }

        #endregion

        #region SettingViewModel -- 设置视图模型

        private SettingViewModel settingViewModel = new SettingViewModel();
        /// <summary>
        /// 设置视图模型
        /// </summary>
        public SettingViewModel SettingViewModel
        {
            get { return settingViewModel; }
            set { settingViewModel = value; this.RaisePropertyChanged(nameof(SettingViewModel)); }
        }

        #endregion

        #region DifferenceValue -- 目标框与中心轴的差值

        private double differenceValue;
        /// <summary>
        /// 目标框与中心轴的差值
        /// </summary>
        public double DifferenceValue
        {
            get { return differenceValue; }
            set { differenceValue = value; this.RaisePropertySaveChanged(nameof(DifferenceValue)); }
        }

        #endregion

        #region AlgorithmDifferenceValue -- 算法目标框与中心轴的偏差

        private double algorithmDifferenceValue;
        /// <summary>
        /// 算法目标框与中心轴的偏差
        /// </summary>
        public double AlgorithmDifferenceValue
        {
            get { return algorithmDifferenceValue; }
            set { algorithmDifferenceValue = value; this.RaisePropertySaveChanged(nameof(AlgorithmDifferenceValue)); }
        }

        #endregion

        #region AlgorithmFPS -- 算法FPS

        private FPSHelper algorithmFPS;
        /// <summary>
        /// 算法FPS
        /// </summary>
        public FPSHelper AlgorithmFPS
        {
            get { return algorithmFPS; }
            set { algorithmFPS = value; this.RaisePropertyChanged(nameof(AlgorithmFPS)); }
        }

        #endregion

        #region AlgorithmCenterX -- 算法中心值

        private double algorithmCenterX;
        /// <summary>
        /// 算法中心值
        /// </summary>
        public double AlgorithmCenterX
        {
            get { return algorithmCenterX; }
            set { algorithmCenterX = value; this.RaisePropertySaveChanged(nameof(AlgorithmCenterX)); }
        }

        #endregion

        // ================================================================================
        // Command
        // ================================================================================

        #region LoadedCommand -- 加载完成命令

        /// <summary>
        /// 加载完成命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载完成
        /// </summary>
        private void Loaded()
        {
            // 初始化视频控件
            this.InitVideoControl();

            // 初始化OpenCV流
            this.InitOpenCvStream();
        }

        /// <summary>
        /// 初始化视频控件
        /// </summary>
        private void InitVideoControl()
        {
            MainView view = this.GetView<MainView>();
            if (view == null)
                return;

            // 调试模式下显示FPS
            view.video.IsShowFPS = ApplicationDomainEx.IS_DEBUG;

            // 跟踪框插件
            TrackingBoxPlugin trackingBoxPlugin = new TrackingBoxPlugin(view.video);
            view.video.AttachPlugin(trackingBoxPlugin);

            // 安全轴插件
            CenterAxisPlugin centerAxisPlugin = new CenterAxisPlugin(view.video);
            view.video.AttachPlugin(centerAxisPlugin);
        }

        /// <summary>
        /// 初始化OpenCV流
        /// </summary>
        private void InitOpenCvStream()
        {
            OpenCVStream stream = VideoStreamManager.Get<OpenCVStream>(VideoViewKeys.Main);
            stream.ExecuteVideoFrame -= Stream_ExecuteVideoFrame;
            stream.ExecuteVideoFrame += Stream_ExecuteVideoFrame;
        }

        /// <summary>
        /// 获取到视频帧时触发
        /// </summary>
        private void Stream_ExecuteVideoFrame(object sender, VideoFrameEventArgs e)
        {
            MainView view = this.GetView<MainView>();
            if (view == null)
                return;

            // 更新视频帧
            view.video.UpdateVideoFrame(e.Frame);

            // 更新安全轴
            CenterAxisInfo centerAxisInfo = new CenterAxisInfo();
            centerAxisInfo.AxisType = CenterAxisType.Vertical;
            centerAxisInfo.SrcCenter = new SharpDX.Mathematics.Interop.RawVector2(this.SettingViewModel.SafeAxis, 0);
            centerAxisInfo.AxisColor = SharpDxColorHelper.FromString("#AA000000");
            centerAxisInfo.AxisWidth = 2;
            view.video.UpdateCenterAxis(centerAxisInfo);
        }

        #endregion

        #region CloseCommand -- 关闭命令

        /// <summary>
        /// 关闭命令
        /// </summary>
        public VCommand CloseCommand { get; set; }

        /// <summary>
        /// 关闭
        /// </summary>
        private void Close()
        {
            MessageBoxExResult result = MessageBoxEx.ShowDialog("提示", "确定退出？", MessageBoxExButtons.YES_CANCEL, this.GetWindow());
            if (result != MessageBoxExResult.YES)
                return;

            this.GetWindow().Close();
        }

        #endregion

        #region MinCommand -- 最小化窗口命令

        /// <summary>
        /// 最小化窗口命令
        /// </summary>
        public VCommand MinCommand { get; set; }

        /// <summary>
        /// 最小化
        /// </summary>
        private void Min()
        {
            this.GetWindow().WindowState = WindowState.Minimized;
        }

        #endregion

        #region SettingCommand --  设置命令

        /// <summary>
        /// 设置命令
        /// </summary>
        public VCommand SettingCommand { get; set; }

        /// <summary>
        /// 设置
        /// </summary>
        private void Setting()
        {
            //SystemSettingView view = new SystemSettingView();
            //NoneWindow window = new NoneWindow(1200, 1000, view);
            //window.Owner = this.GetWindow();

            //window.ShowDialog();
        }

        #endregion

        // ================================================================================
        // Message
        // ================================================================================

        /// <summary>
        /// 处理目标检测消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnAlgorithmDectectMessage(AlgorithmDectectMessage msg)
        {
            MainView view = this.GetView<MainView>();
            if (view == null)
                return;

            // 如果没有目标框，或目标框数据不正确，那么清理目标框
            if (msg.target_bbox == null || msg.target_bbox.Count != 4)
            {
                view.video.ClearTrackingBox();

                // 更新FPS
                this.AlgorithmFPS.CalcFps();

                return;
            }

            // 更新目标框位置
            List<TrackingBoxInfo> infos = new List<TrackingBoxInfo>();
            TrackingBoxInfo info = new TrackingBoxInfo();
            info.DrawingBorderWidth = 2;
            info.DrawingBorderColor = SharpDxColorHelper.FromString("#FFFF0000");

            float left = msg.target_bbox[0];
            float top = msg.target_bbox[1];
            float right = msg.target_bbox[2];
            float bottom = msg.target_bbox[3];

            info.SrcRect = new SharpDX.Mathematics.Interop.RawRectangleF(left, top, right, bottom);

            infos.Add(info);

            view.video.UpdateTrackingBox(infos);

            // 更新差值
            double center = left + (right - left) / 2d;
            this.DifferenceValue = center - this.SettingViewModel.SafeAxis;
            this.AlgorithmCenterX = msg.center_x - this.SettingViewModel.SafeAxis;
            this.AlgorithmDifferenceValue = this.SettingViewModel.SafeAxis - msg.center_x;

            // 更新FPS
            this.AlgorithmFPS.CalcFps();
        }
    }
}
