﻿using SharpDX.Direct2D1;
using SharpDX.DirectWrite;
using SharpDX.Mathematics.Interop;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using VIZ.Framework.Core;
using VIZ.Framework.Domain;

namespace VIZ.Framework.Common
{
    /// <summary>
    /// 跟踪框插件
    /// </summary>
    public class TrackingBoxPlugin : VideoPluginBase
    {
        /// <summary>
        /// 跟踪框插件
        /// </summary>
        /// <param name="videoControl">视频控件</param>
        public TrackingBoxPlugin(VideoControl videoControl) : base(videoControl)
        {

        }

        /// <summary>
        /// 名称
        /// </summary>
        public override string Name => VideoControlPluginNames.TrackingBox;

        /// <summary>
        /// 点击跟踪框时触发
        /// </summary>
        public event EventHandler<TrackingBoxClickEventArgs> TrackingBoxClick;

        /// <summary>
        /// 跟踪框信息
        /// </summary>
        private volatile List<TrackingBoxInfo> trackingBoxInfos;

        /// <summary>
        /// 格式
        /// </summary>
        private TextFormat textFormat;

        /// <summary>
        /// 更新数据
        /// </summary>
        /// <param name="trackingBoxInfos">跟踪框信息</param>
        public void Update(List<TrackingBoxInfo> trackingBoxInfos)
        {
            this.trackingBoxInfos = trackingBoxInfos;
        }

        /// <summary>
        /// 渲染
        /// </summary>
        /// <param name="context">渲染上下文</param>
        public override void Render(VideoRenderContext context)
        {
            // 画面渲染区域
            List<TrackingBoxInfo> infos = this.trackingBoxInfos;
            if (infos == null)
                return;

            foreach (TrackingBoxInfo info in infos)
            {
                // 跟踪框绘制区域转化为UI绘制区域
                SharpDX.Mathematics.Interop.RawRectangleF rect;
                if (context.Mode == VideoRenderMode.UI)
                {
                    Rect drawRect = ImageHelper.ConvertImageRectToUiRect(context.VideoRenderInfo.Frame.Width, context.VideoRenderInfo.Frame.Height, context.VideoRenderInfo.DrawingRect, info.SrcRect.ToRect());
                    rect = drawRect.ToRawRectangleF();
                }
                else
                {
                    rect = info.SrcRect;
                }

                // 绘制跟踪框
                SolidColorBrush brush = new SolidColorBrush(context.Target, info.DrawingBorderColor);
                context.Target.DrawRectangle(rect, brush, info.DrawingBorderWidth);

                // 在调试模式下绘制目标框面积占比
                if (!ApplicationDomain.IS_DEBUG)
                    continue;

                double all = context.VideoRenderInfo.Frame.Width * context.VideoRenderInfo.Frame.Height;
                double use = (info.SrcRect.Right - info.SrcRect.Left) * (info.SrcRect.Bottom - info.SrcRect.Top);
                double p = use / all;

                if (this.textFormat == null)
                {
                    this.textFormat = new TextFormat(context.DirectWriteFactory, "Microsoft YaHei", 14);
                }

                TextLayout textLayout = new TextLayout(context.DirectWriteFactory, p.ToString("P2"), this.textFormat, 100, 20);
                context.Target.DrawTextLayout(new RawVector2(rect.Left, rect.Top - 20), textLayout, brush);
            }
        }

        /// <summary>
        /// 附加
        /// </summary>
        public override void Attach()
        {
            this.VideoRender.PreviewMouseLeftButtonDown -= VideoRender_PreviewMouseLeftButtonDown;
            this.VideoRender.PreviewMouseLeftButtonDown += VideoRender_PreviewMouseLeftButtonDown;
        }

        /// <summary>
        /// 卸载
        /// </summary>
        public override void Detach()
        {
            this.VideoRender.PreviewMouseLeftButtonDown -= VideoRender_PreviewMouseLeftButtonDown;
        }

        /// <summary>
        /// 鼠标左键点击
        /// </summary>
        private void VideoRender_PreviewMouseLeftButtonDown(object sender, System.Windows.Input.MouseButtonEventArgs e)
        {
            if (this.TrackingBoxClick == null || !this.IsEnabled)
                return;

            VideoRenderInfo renderInfo = this.VideoRender.RenderInfo;
            if (renderInfo == null)
                return;

            Point uiPoint = e.GetPosition(this.VideoRender);
            Point bmpPoint = ImageHelper.ConvertUiPointToImagePoint(renderInfo.Frame.Width, renderInfo.Frame.Height, renderInfo.DrawingRect, uiPoint);

            TrackingBoxClickEventArgs args = new TrackingBoxClickEventArgs();
            args.MousePoint = uiPoint;
            args.HitTrackingBoxInfo = TrackingBoxExpand.HitTest(this.trackingBoxInfos, bmpPoint);

            if (args.HitTrackingBoxInfo == null)
                return;

            // 触发视频矩形框点击事件
            this.TrackingBoxClick.Invoke(this, args);
        }
    }
}
