﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using SharpDX.Mathematics.Interop;

namespace VIZ.Framework.Common
{
    /// <summary>
    /// 视频流基类
    /// </summary>
    /// <typeparam name="TOption">设置类型</typeparam>
    public abstract class VideoStreamBase<TOption> : IVideoStream, IDisposable where TOption : VideoStreamOptionBase
    {
        /// <summary>
        /// 视频流基类
        /// </summary>
        /// <param name="option">设置</param>
        public VideoStreamBase(TOption option)
        {
            this.Option = option;
        }

        /* ========================================================================================================= */
        /* === Const === */
        /* ========================================================================================================= */

        /// <summary>
        /// 视频帧队列最大容量
        /// 如果超过该容量，那么会抛弃对队列头部的处理
        /// </summary>
        public const int VIDEO_FRAME_QUEUE_MAX_COUNT = 8;

        /* ========================================================================================================= */
        /* === Event === */
        /* ========================================================================================================= */

        /// <summary>
        /// 处理视频帧事件
        /// </summary>
        public event EventHandler<VideoFrameEventArgs> ExecuteVideoFrame;

        /* ========================================================================================================= */
        /* === Property === */
        /* ========================================================================================================= */

        /// <summary>
        /// 设置
        /// </summary>
        public TOption Option { get; private set; }

        /* ========================================================================================================= */
        /* === Internal Field === */
        /* ========================================================================================================= */

        /// <summary>
        /// 视频帧队列
        /// </summary>
        internal ConcurrentQueue<IVideoFrame> VideoFrameQueue = new ConcurrentQueue<IVideoFrame>();

        /* ========================================================================================================= */
        /* === Internal Function === */
        /* ========================================================================================================= */

        /// <summary>
        /// 触发执行视频帧事件
        /// </summary>
        /// <param name="frame">视频帧</param>
        internal void TriggerExecuteVideoFrame(IVideoFrame frame)
        {
            if (this.ExecuteVideoFrame == null)
                return;

            VideoFrameEventArgs args = new VideoFrameEventArgs();
            args.Frame = frame;

            this.ExecuteVideoFrame.Invoke(this, args);
        }

        /* ========================================================================================================= */
        /* === Public Function === */
        /* ========================================================================================================= */

        /// <summary>
        /// 销毁
        /// </summary>
        public abstract void Dispose();

        /// <summary>
        /// 设置视频帧延时
        /// </summary>
        /// <param name="delayFrame">延时帧</param>
        public void SetVideoFrameDelay(int delayFrame)
        {
            this.Option.DelayFrame = delayFrame;
            while (this.VideoFrameQueue.Count > VIDEO_FRAME_QUEUE_MAX_COUNT + this.Option.DelayFrame)
            {
                this.VideoFrameQueue.TryDequeue(out IVideoFrame remove);
                remove?.Dispose();
            }
        }

        /* ========================================================================================================= */
        /* === Protected Function === */
        /* ========================================================================================================= */
    }
}
