﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using VIZ.Framework.Core;
using OpenCvSharp;
using OpenCvSharp.WpfExtensions;

namespace VIZ.Framework.ImageContrastTool
{
    /// <summary>
    /// 主视图模型
    /// </summary>
    public class MainViewModel : ViewModelBase
    {
        public MainViewModel()
        {
            // 初始化命令
            this.InitCommand();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
            this.SelectLeftPathCommand = new VCommand(this.SelectLeftPath);
            this.SelectRightPathCommand = new VCommand(this.SelectRightPath);
            this.RefreshCommand = new VCommand(this.Refresh);
            this.NextCommand = new VCommand(this.Next);
            this.PrevCommand = new VCommand(this.Prev);
        }

        // --------------------------------------------------------------------
        // Field
        // --------------------------------------------------------------------

        /// <summary>
        /// 系统配置
        /// </summary>
        private SystemConfig systemConfig;

        // --------------------------------------------------------------------
        // Property
        // --------------------------------------------------------------------

        #region LeftPath -- 左侧图片文件夹路径

        private string leftPath;
        /// <summary>
        /// 左侧图片文件夹路径
        /// </summary>
        public string LeftPath
        {
            get { return leftPath; }
            set { leftPath = value; this.RaisePropertyChanged(nameof(LeftPath)); }
        }

        #endregion

        #region RightPath -- 右侧图片文件夹路径

        private string rightPath;
        /// <summary>
        /// 右侧图片文件夹路径
        /// </summary>
        public string RightPath
        {
            get { return rightPath; }
            set { rightPath = value; this.RaisePropertyChanged(nameof(RightPath)); }
        }

        #endregion

        #region ImageFiles -- 图片文件集合

        private List<string> imageFiles;
        /// <summary>
        /// 图片文件集合
        /// </summary>
        public List<string> ImageFiles
        {
            get { return imageFiles; }
            set { imageFiles = value; this.RaisePropertyChanged(nameof(ImageFiles)); }
        }

        #endregion

        #region SelectedImageFile -- 当前选中的图片值

        private string selectedImageFile;
        /// <summary>
        /// 当前选中的图片值
        /// </summary>
        public string SelectedImageFile
        {
            get { return selectedImageFile; }
            set 
            {
                selectedImageFile = value;
                this.RaisePropertyChanged(nameof(SelectedImageFile));
                this.UpdateRightImageFile();
            }
        }

        #endregion

        #region RightImageFile -- 右侧图片路径

        private string rightImageFile;
        /// <summary>
        /// 右侧图片路径
        /// </summary>
        public string RightImageFile
        {
            get { return rightImageFile; }
            set { rightImageFile = value; this.RaisePropertyChanged(nameof(RightImageFile)); }
        }

        #endregion

        // --------------------------------------------------------------------
        // Command
        // --------------------------------------------------------------------

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {
            this.systemConfig = ApplicationDomainEx.LiteDbContext.SystemConfigs.FindAll().FirstOrDefault();

            this.LeftPath = this.systemConfig.LeftPath;
            this.RightPath = this.systemConfig.RightPath;
        }

        #endregion

        #region SelectLeftPathCommand -- 选择左侧图片文件夹路径命令

        /// <summary>
        /// 选择左侧图片文件夹路径命令
        /// </summary>
        public VCommand SelectLeftPathCommand { get; set; }

        /// <summary>
        /// 选择左侧图片文件夹路径
        /// </summary>
        private void SelectLeftPath()
        {
            FolderBrowserDialog fbd = new FolderBrowserDialog();
            if (fbd.ShowDialog() != DialogResult.OK)
                return;

            this.LeftPath = fbd.SelectedPath;

            this.systemConfig.LeftPath = this.LeftPath;
            ApplicationDomainEx.LiteDbContext.SystemConfigs.Update(this.systemConfig);
        }

        #endregion

        #region SelectRightPathCommand -- 选择右侧图片文件夹路径命令

        /// <summary>
        /// 选择右侧图片文件夹路径命令
        /// </summary>
        public VCommand SelectRightPathCommand { get; set; }

        /// <summary>
        /// 选择右侧图片文件夹路径
        /// </summary>
        private void SelectRightPath()
        {

            FolderBrowserDialog fbd = new FolderBrowserDialog();
            if (fbd.ShowDialog() != DialogResult.OK)
                return;

            this.RightPath = fbd.SelectedPath;

            this.systemConfig.RightPath = this.RightPath;
            ApplicationDomainEx.LiteDbContext.SystemConfigs.Update(this.systemConfig);
        }

        #endregion

        #region RefreshCommand -- 刷新命令

        /// <summary>
        /// 刷新命令
        /// </summary>
        public VCommand RefreshCommand { get; set; }

        /// <summary>
        /// 刷新
        /// </summary>
        private void Refresh()
        {
            if (System.IO.Directory.Exists(this.LeftPath))
            {
                this.ImageFiles = System.IO.Directory.GetFiles(this.LeftPath, "*.*").ToList();
            }
            else
            {
                this.ImageFiles = null;
            }

            this.SelectedImageFile = null;
            this.RightImageFile = null;
        }

        #endregion

        #region NextCommand -- 下一张命令

        /// <summary>
        /// 下一张命令
        /// </summary>
        public VCommand NextCommand { get; set; }

        /// <summary>
        /// 下一张
        /// </summary>
        private void Next()
        {
            if (this.SelectedImageFile == null)
            {
                this.SelectedImageFile = this.ImageFiles.FirstOrDefault();
            }
            else
            {
                int index = this.ImageFiles.IndexOf(this.SelectedImageFile);
                ++index;
                if (index <= this.ImageFiles.Count - 1)
                {
                    this.SelectedImageFile = this.ImageFiles[index];
                }
            }
        }

        #endregion

        #region PrevCommand -- 上一张命令

        /// <summary>
        /// 上一张命令
        /// </summary>
        public VCommand PrevCommand { get; set; }

        /// <summary>
        /// 上一张
        /// </summary>
        private void Prev()
        {
            if (this.SelectedImageFile == null)
            {
                this.SelectedImageFile = this.ImageFiles.FirstOrDefault();
            }
            else
            {
                int index = this.ImageFiles.IndexOf(this.SelectedImageFile);
                --index;
                if (index >= 0)
                {
                    this.SelectedImageFile = this.ImageFiles[index];
                }
            }
        }

        #endregion

        // --------------------------------------------------------------------
        // Private Function
        // --------------------------------------------------------------------

        /// <summary>
        /// 更新右侧图片路径
        /// </summary>
        private void UpdateRightImageFile()
        {
            if (string.IsNullOrWhiteSpace(this.SelectedImageFile))
            {
                this.RightImageFile = null;
                return;
            }

            string fileName = System.IO.Path.GetFileName(this.SelectedImageFile);
            this.RightImageFile = System.IO.Path.Combine(this.RightPath, fileName);
        }
    }
}
