﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using Newtonsoft.Json;
using VIZ.Framework.Connection;
using VIZ.Framework.Core;
using VIZ.GimbalAI.Controller.Connection;
using VIZ.GimbalAI.Controller.Domain;
using VIZ.GimbalAI.Controller.Storage;

namespace VIZ.GimbalAI.Controller.Module
{
    /// <summary>
    /// 云台信号保存控制器
    /// </summary>
    public class GimbalSignalSaveController
    {
        /// <summary>
        /// 云台信号保存控制器
        /// </summary>
        /// <param name="support">支持</param>
        public GimbalSignalSaveController(IGimbalSignalSaveSupport support)
        {
            this.Support = support;
        }

        /// <summary>
        /// 支持
        /// </summary>
        public IGimbalSignalSaveSupport Support { get; private set; }

        /// <summary>
        /// 消息队列
        /// </summary>
        public ConcurrentQueue<ConnFixedBufferMessage> MessageQueue { get; private set; } = new ConcurrentQueue<ConnFixedBufferMessage>();

        /// <summary>
        /// 是否正在执行
        /// </summary>
        public bool IsRunning { get; private set; }

        /// <summary>
        /// 是否输出日志
        /// </summary>
        private static readonly bool GIMBAL_IS_WRITE_DATA = ApplicationDomainEx.IniStorage.GetValue<GimbalConfig, bool>(p => p.GIMBAL_IS_WRITE_DATA);

        /// <summary>
        /// 处理线程
        /// </summary>
        private System.Threading.Thread thread;

        /// <summary>
        /// 文件流
        /// </summary>
        private FileStream fileStream;

        /// <summary>
        /// 启动
        /// </summary>
        public void Start()
        {
            if (GIMBAL_IS_WRITE_DATA)
            {
                string floder = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "data");
                if (!Directory.Exists(floder))
                {
                    Directory.CreateDirectory(floder);
                }

                string path = Path.Combine(floder, $"data_{DateTime.Now.ToString("yyyy_MM_dd___HH_mm_ss")}.json");
                this.fileStream = new FileStream(path, FileMode.Create, FileAccess.Write);
            }

            this.IsRunning = true;
            this.thread = new System.Threading.Thread(this.Execute);
            this.thread.IsBackground = true;

            this.thread.Start();
        }

        /// <summary>
        /// 停止
        /// </summary>
        public void Stop()
        {
            this.fileStream?.Flush();
            this.fileStream?.Close();
            this.fileStream?.Dispose();
            this.fileStream = null;
            this.IsRunning = false;
            this.thread = null;
        }

        /// <summary>
        /// 执行
        /// </summary>
        private void Execute()
        {
            while (this.IsRunning)
            {
                System.Threading.Thread.Sleep(100);

                bool isUpdateUI_down = false;
                bool isUpdateUI_up = false;

                while (this.MessageQueue.Count > 0)
                {
                    if (!this.MessageQueue.TryDequeue(out ConnFixedBufferMessage msg))
                        continue;

                    // 记录日志
                    GimbalPackage_down down;
                    GimbalPackage_up up;

                    this.Save(msg, out down, out up);

                    // 更新界面
                    if (!isUpdateUI_down && down != null)
                    {
                        isUpdateUI_down = true;

                        WPFHelper.BeginInvoke(() =>
                        {
                            this.Support.DownData = down;
                            this.Support.CurrentCommandStatus = ApplicationDomainEx.CurrentGimbalCommandStatus;
                        });
                    }

                    if (!isUpdateUI_up && up != null)
                    {
                        isUpdateUI_up = true;

                        WPFHelper.BeginInvoke(() =>
                        {
                            this.Support.UpData = up;
                            this.Support.CurrentCommandStatus = ApplicationDomainEx.CurrentGimbalCommandStatus;
                        });
                    }
                }

                this.fileStream?.Flush();
            }
        }

        /// <summary>
        /// 保存消息
        /// </summary>
        /// <param name="msg"></param>
        private void Save(ConnFixedBufferMessage msg, out GimbalPackage_down down, out GimbalPackage_up up)
        {
            down = null;
            up = null;

            if (msg.Provider is GimbalPackageDownProvider)
            {
                // 下行数据
                GimbalPackage_down data = new GimbalPackage_down();
                data.FromBuffer(msg.Buffer);
                data.DateTime = msg.DateTime.ToString("yyyy-MM-dd HH:mm:ss fff");

                if (GIMBAL_IS_WRITE_DATA)
                {
                    string json = JsonConvert.SerializeObject(data);
                    json += "\n";
                    byte[] buffer = Encoding.UTF8.GetBytes(json);

                    this.fileStream?.Write(buffer, 0, buffer.Length);
                }

                down = data;
            }
            else if (msg.Provider is GimbalPackageUpProvider)
            {
                // 上行数据
                GimbalPackage_up data = new GimbalPackage_up();
                data.FromBuffer(msg.Buffer);
                data.DateTime = msg.DateTime.ToString("yyyy-MM-dd HH:mm:ss fff");

                if (GIMBAL_IS_WRITE_DATA)
                {
                    string json = JsonConvert.SerializeObject(data);
                    json += "\n";
                    byte[] buffer = Encoding.UTF8.GetBytes(json);

                    this.fileStream?.Write(buffer, 0, buffer.Length);
                }

                up = data;
            }
        }
    }
}
