﻿using DevExpress.Xpf.Core;
using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using VIZ.Framework.Common;
using VIZ.Framework.Core;
using VIZ.Package.Domain;
using VIZ.Package.Plugin;
using VIZ.Package.Service;
using VIZ.Package.Storage;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 设置窗口模型
    /// </summary>
    public class SettingWindowModel : ViewModelBase
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(SettingWindowModel));

        public SettingWindowModel()
        {
            // 初始化命令
            this.InitCommand();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
            this.SaveCommand = new VCommand(this.Save);
            this.CancelCommand = new VCommand(this.Cancel);
        }

        // ==========================================================================
        // Controller & Service
        // ==========================================================================

        /// <summary>
        /// 操作日志服务
        /// </summary>
        private RecordLogService recordLogService = new RecordLogService();

        // ==========================================================================
        // Property
        // ==========================================================================

        #region ItemsSource -- 数据源

        private ObservableCollection<SettingPageModel> itemsSource = new ObservableCollection<SettingPageModel>();
        /// <summary>
        /// 数据源
        /// </summary>
        public ObservableCollection<SettingPageModel> ItemsSource
        {
            get { return itemsSource; }
            set { itemsSource = value; this.RaisePropertyChanged(nameof(ItemsSource)); }
        }

        #endregion

        #region SelectedValue -- 当前选中的设置项

        private SettingPageModel selectedValue;
        /// <summary>
        /// 当前选中的设置项
        /// </summary>
        public SettingPageModel SelectedValue
        {
            get { return selectedValue; }
            set { selectedValue = value; this.RaisePropertyChanged(nameof(SelectedValue)); }
        }

        #endregion

        // ==========================================================================
        // Command
        // ==========================================================================

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {
            if (this.IsAlreadyLoaded)
                return;

            this.IsAlreadyLoaded = true;

            var settings = ApplicationDomainEx.PluginInfos.Where(p => p.PluginType == PluginType.Setting &&
                                                                      !string.IsNullOrWhiteSpace(p.Group) &&
                                                                      (p.Group == ApplicationConstants.APPLICATION_GROUP_NAME || p.Group == ApplicationDomainEx.VizConfig.PluginGroup));

            foreach (var item in settings)
            {
                SettingPageModel config = new SettingPageModel();
                config.Name = item.Name;
                config.ViewType = item.ViewType;
                this.ItemsSource.Add(config);
            }

            this.SelectedValue = this.ItemsSource.FirstOrDefault();
        }

        #endregion

        #region SaveCommand -- 保存命令

        /// <summary>
        /// 保存命令
        /// </summary>
        public VCommand SaveCommand { get; set; }

        /// <summary>
        /// 保存
        /// </summary>
        private void Save()
        {
            if (this.SelectedValue == null || this.SelectedValue.View == null)
                return;

            if (!this.SelectedValue.View.TryGetTarget(out object target))
                return;

            if (!(target is IPluginSettingView view))
                return;

            try
            {
                view.Save();

                // 记录操作日志
                string remark = $"{this.SelectedValue.Name}";
                this.recordLogService.AppendLog(ApplicationConstants.APPLICATION_GROUP_NAME, RecordLogOperate.Operate, RecordLogTrigger.Human, RecordLogConstants.OPERATE_SETTING_SAVE, remark);

                DXMessageBox.Show("保存成功!");
            }
            catch (Exception ex)
            {
                log.Error(ex);

                DXMessageBox.Show("保存失败!");
            }
        }

        #endregion

        #region CancelCommand -- 取消命令

        /// <summary>
        /// 取消命令
        /// </summary>
        public VCommand CancelCommand { get; set; }

        /// <summary>
        /// 取消
        /// </summary>
        private void Cancel()
        {
            Window window = this.GetWindow();
            window?.Close();
        }

        #endregion
    }
}
