﻿using DevExpress.Xpf.Editors;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Input;
using System.Windows;

namespace VIZ.Package.Common
{
    /// <summary>
    /// 左右拖拽改变值文本编辑器
    /// </summary>
    public class LeftRightTextEdit : TextEdit
    {
        #region MouseMinMove -- 鼠标最小移动值

        /// <summary>
        /// 鼠标最小移动值
        /// </summary>
        public double MouseMinMove
        {
            get { return (double)GetValue(MouseMinMoveProperty); }
            set { SetValue(MouseMinMoveProperty, value); }
        }

        /// <summary>
        /// Using a DependencyProperty as the backing store for MouseMinMove.  This enables animation, styling, binding, etc...
        /// </summary>
        public static readonly DependencyProperty MouseMinMoveProperty =
            DependencyProperty.Register("MouseMinMove", typeof(double), typeof(LeftRightTextEdit), new PropertyMetadata(5d));

        #endregion

        #region ValueChangeStep -- 值改变步长

        /// <summary>
        /// 值改变步长
        /// </summary>
        public double ValueChangeStep
        {
            get { return (double)GetValue(ValueChangeStepProperty); }
            set { SetValue(ValueChangeStepProperty, value); }
        }

        /// <summary>
        /// Using a DependencyProperty as the backing store for ValueChangeStep.  This enables animation, styling, binding, etc...
        /// </summary>
        public static readonly DependencyProperty ValueChangeStepProperty =
            DependencyProperty.Register("ValueChangeStep", typeof(double), typeof(LeftRightTextEdit), new PropertyMetadata(0.1d));

        #endregion

        /// <summary>
        /// 鼠标是否按下
        /// </summary>
        private bool isMouseDown;

        /// <summary>
        /// 鼠标最后的值
        /// </summary>
        private double mouseLastPositionX;

        /// <summary>
        /// 之前的鼠标形状
        /// </summary>
        private Cursor beforeCursor;

        /// <summary>
        /// 鼠标按下
        /// </summary>
        protected override void OnPreviewMouseLeftButtonDown(MouseButtonEventArgs e)
        {
            base.OnPreviewMouseLeftButtonDown(e);
            this.mouseLastPositionX = e.GetPosition(this).X;
            this.beforeCursor = this.Cursor;
            this.Cursor = Cursors.ScrollWE;
            this.isMouseDown = true;
        }

        /// <summary>
        /// 鼠标弹起
        /// </summary>
        protected override void OnPreviewMouseLeftButtonUp(MouseButtonEventArgs e)
        {
            base.OnPreviewMouseLeftButtonUp(e);
            this.Cursor = this.beforeCursor;
            this.beforeCursor = null;
            this.isMouseDown = false;
        }

        /// <summary>
        /// 鼠标移动
        /// </summary>
        protected override void OnPreviewMouseMove(MouseEventArgs e)
        {
            base.OnPreviewMouseMove(e);
            if (!this.isMouseDown || e.LeftButton != MouseButtonState.Pressed)
                return;

            Point position = e.GetPosition(this);

            double move = position.X - this.mouseLastPositionX;
            if (Math.Abs(move) < this.MouseMinMove)
                return;

            double.TryParse(this.EditValue?.ToString(), out double current);

            this.EditValue = current + (int)(move / this.MouseMinMove) * this.ValueChangeStep;
            this.mouseLastPositionX = position.X;
        }
    }
}