﻿using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.Package.Connection;
using VIZ.Package.Domain;
using VIZ.Package.Service;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 连接设置视图模型
    /// </summary>
    public class ConnSettingViewModel : ViewModelBase
    {
        /// <summary>
        /// 日志
        /// </summary>
        private static readonly ILog log = LogManager.GetLogger(typeof(ConnSettingViewModel));

        public ConnSettingViewModel()
        {
            // 初始化命令
            this.InitCommand();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
            this.AddGroupCommand = new VCommand(this.AddGroup);
            this.DeleteGroupCommand = new VCommand(this.DeleteGroup);
            this.SetDefaultGroupCommand = new VCommand(this.SetDefaultGroup);
            this.AddItemCommand = new VCommand(this.AddItem);
            this.DeleteItemCommand = new VCommand(this.DeleteItem);
        }

        // ==========================================================================
        // Field & Service
        // ==========================================================================

        /// <summary>
        /// 连接服务
        /// </summary>
        private ConnService connService = new ConnService();

        // ======================================================================================
        // Property
        // ======================================================================================

        #region Groups -- 分组集合

        private ObservableCollection<ConnGroupModel> groups;
        /// <summary>
        /// 分组集合
        /// </summary>
        public ObservableCollection<ConnGroupModel> Groups
        {
            get { return groups; }
            set { groups = value; this.RaisePropertyChanged(nameof(Groups)); }
        }

        #endregion

        #region SelectedGroup -- 当前选中的分组

        private ConnGroupModel selectedGroup;
        /// <summary>
        /// 当前选中的分组
        /// </summary>
        public ConnGroupModel SelectedGroup
        {
            get { return selectedGroup; }
            set { selectedGroup = value; this.RaisePropertyChanged(nameof(SelectedGroup)); }
        }

        #endregion

        #region SelectedItem -- 当前选中的项

        private ConnModel selectedItem;
        /// <summary>
        /// 当前选中的项
        /// </summary>
        public ConnModel SelectedItem
        {
            get { return selectedItem; }
            set { selectedItem = value; this.RaisePropertyChanged(nameof(SelectedItem)); }
        }

        #endregion

        // ======================================================================================
        // Command
        // ======================================================================================

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {
            if (this.IsAlreadyLoaded)
                return;

            this.IsAlreadyLoaded = true;

            this.Groups = this.connService.LoadGroups().ToObservableCollection();
        }

        #endregion

        #region AddGroupCommand -- 添加分组命令

        /// <summary>
        /// 添加分组命令
        /// </summary>
        public VCommand AddGroupCommand { get; set; }

        /// <summary>
        /// 添加分组
        /// </summary>
        private void AddGroup()
        {
            ConnGroupModel group = new ConnGroupModel();
            group.GroupID = Guid.NewGuid();
            group.Name = "新建分组";
            this.Groups.Add(group);
        }

        #endregion

        #region DeleteGroupCommand -- 删除分组命令

        /// <summary>
        /// 删除分组
        /// </summary>
        public VCommand DeleteGroupCommand { get; set; }

        /// <summary>
        /// 删除分组
        /// </summary>
        private void DeleteGroup()
        {
            if (this.SelectedGroup == null)
                return;

            this.Groups.Remove(this.SelectedGroup);
        }

        #endregion

        #region SetDefaultGroupCommand -- 启用组命令

        /// <summary>
        /// 设置默认分组命令
        /// </summary>
        public VCommand SetDefaultGroupCommand { get; set; }

        /// <summary>
        /// 设置默认分组
        /// </summary>
        private void SetDefaultGroup()
        {
            if (this.SelectedGroup == null)
                return;

            foreach (var group in this.Groups)
            {
                group.IsDefault = group == this.SelectedGroup;
            }
        }

        #endregion

        #region AddItemCommand -- 添加项命令

        /// <summary>
        /// 添加项命令
        /// </summary>
        public VCommand AddItemCommand { get; set; }

        /// <summary>
        /// 添加项
        /// </summary>
        private void AddItem()
        {
            if (this.SelectedGroup == null)
                return;

            ConnModel item = new ConnModel();
            item.ConnID = Guid.NewGuid();
            item.GroupID = this.SelectedGroup.GroupID;
            item.IP = "127.0.0.1";
            item.Port = 6100;

            this.SelectedGroup.Items.Add(item);
        }

        #endregion

        #region DeleteItemCommand -- 删除项命令

        /// <summary>
        /// 删除项命令
        /// </summary>
        public VCommand DeleteItemCommand { get; set; }

        /// <summary>
        /// 删除项
        /// </summary>
        private void DeleteItem()
        {
            if (this.SelectedItem == null || this.SelectedGroup == null)
                return;

            this.SelectedGroup.Items.Remove(this.SelectedItem);
        }

        #endregion

        // ======================================================================================
        // Public Function
        // ======================================================================================

        /// <summary>
        /// 保存
        /// </summary>
        public void Save()
        {
            // 保存配置
            this.connService.SaveGroups(this.Groups);
            ApplicationDomainEx.ConnGroups.Clear();
            ApplicationDomainEx.ConnGroups.AddRange(this.Groups.ToList());

            // 发送连接改变消息
            ConnChangedMessage msg = new ConnChangedMessage();
            ApplicationDomainEx.MessageManager.Send(msg);
        }
    }
}
