﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.Package.Domain;
using VIZ.Package.Plugin;
using VIZ.Package.Storage;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 主视图模型
    /// </summary>
    public class MainViewModel : ViewModelBase, IMainViewService
    {
        public MainViewModel()
        {
            // 初始化命令
            this.InitCommand();

            // 初始化消息
            this.InitMessage();

            // 注册服务
            ApplicationDomainEx.ServiceManager.AddService(ViewServiceKeys.MAIN_VIEW_SERVICE, this);
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void InitMessage()
        {
            ApplicationDomainEx.MessageManager.Register<ApplicationClosedMessage>(this, this.OnApplicationClosedMessage);
        }

        // ============================================================
        // Service & Controller
        // ============================================================

        // ============================================================
        // Property
        // ============================================================

        #region ItemsSource -- 视图项

        private ObservableCollection<PluginInfo> itemsSource;
        /// <summary>
        /// 视图项
        /// </summary>
        public ObservableCollection<PluginInfo> ItemsSource
        {
            get { return itemsSource; }
            set { itemsSource = value; this.RaisePropertyChanged(nameof(ItemsSource)); }
        }

        #endregion

        // ============================================================
        // Command
        // ============================================================

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {
            if (this.IsAlreadyLoaded)
                return;

            this.IsAlreadyLoaded = true;

            this.ItemsSource = ApplicationDomainEx.PluginInfos.Where(p => (p.Group == ApplicationConstants.APPLICATION_GROUP_NAME ||
                                                                           p.Group == ApplicationDomainEx.VizConfig.PluginGroup) &&
                                                                           p.PluginType == PluginType.Module).ToObservableCollection();

            // 加载布局
            this.LoadLayout();

            // 创建VizCommand窗口
            ApplicationDomainEx.VizCommandLogWindow = new VizCommandWindow();
            ApplicationDomainEx.VizCommandLogWindow.Owner = ApplicationDomainEx.MainWindow;

            // 创建Error窗口
            ApplicationDomainEx.ErrorLogWindow = new ErrorLogWindow();
            ApplicationDomainEx.ErrorLogWindow.Owner = ApplicationDomainEx.MainWindow;
        }

        #endregion

        // ============================================================
        // Message
        // ============================================================

        /// <summary>
        /// 系统关闭消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnApplicationClosedMessage(ApplicationClosedMessage msg)
        {
            // 保存布局
            this.SaveLayout();
        }

        // ============================================================
        // Public Function
        // ============================================================

        /// <summary>
        /// 保存布局
        /// </summary>
        public void SaveLayout()
        {
            MainView view = this.GetView<MainView>();
            if (view == null)
                return;

            string path = this.GetLayoutPath();

            // 创建布局文件文件夹
            string dir = System.IO.Path.GetDirectoryName(path);
            if (!System.IO.Directory.Exists(dir))
            {
                System.IO.Directory.CreateDirectory(dir);
            }

            view.dockLayoutManager.SaveLayoutToXml(path);
        }

        /// <summary>
        /// 加载布局
        /// </summary>
        public void LoadLayout()
        {
            MainView view = this.GetView<MainView>();
            if (view == null)
                return;

            // 获取当前布局文件路径
            string path = this.GetLayoutPath();

            // 如果没有布局文件，那么尝试使用默认布局文件
            if (string.IsNullOrWhiteSpace(path) || !System.IO.File.Exists(path))
            {
                path = this.GetDefaultLayoutPath();
            }

            // 如果没有对应插件布局文件，那么尝试使用系统默认布局文件
            if (string.IsNullOrWhiteSpace(path) || !System.IO.File.Exists(path))
            {
                path = System.IO.Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "layout", "default_layout.xml");
            }

            // 如果布局文件不存在，那么不加载布局
            if (!System.IO.File.Exists(path))
            {
                return;
            }

            view.dockLayoutManager.RestoreLayoutFromXml(path);
        }

        // ============================================================
        // Private Function
        // ============================================================

        /// <summary>
        /// 获取布局文件路径
        /// </summary>
        /// <returns>布局文件路径</returns>
        private string GetLayoutPath()
        {
            string path;

            if (string.IsNullOrWhiteSpace(ApplicationDomainEx.VizConfig.PluginGroup))
            {
                path = System.IO.Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "layout", "layout.xml");
            }
            else
            {
                path = System.IO.Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "layout", ApplicationDomainEx.VizConfig.PluginGroup, "layout.xml");
            }

            return path;
        }

        /// <summary>
        /// 获取默认布局文件路径
        /// </summary>
        /// <returns>默认布局文件路径</returns>
        private string GetDefaultLayoutPath()
        {
            string path;

            if (string.IsNullOrWhiteSpace(ApplicationDomainEx.VizConfig.PluginGroup))
            {
                path = System.IO.Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "layout", "default_layout.xml");
            }
            else
            {
                path = System.IO.Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "layout", ApplicationDomainEx.VizConfig.PluginGroup, "default_layout.xml");
            }

            return path;
        }
    }
}
