﻿using DevExpress.Xpf.Core;
using LiteDB;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using VIZ.Framework.Common;
using VIZ.Framework.Core;
using VIZ.Package.Domain;
using VIZ.Package.Service;
using VIZ.Package.Storage;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 字段编辑视图模型
    /// </summary>
    public class FieldEditViewModel : FieldEditViewModelBase, IFieldEditService
    {
        public FieldEditViewModel()
        {
            // 初始化命令
            this.InitCommand();

            // 初始化消息
            this.InitMessage();

            // 注册服务
            ApplicationDomainEx.ServiceManager.AddService(ViewServiceKeys.FIELD_EDIT_SERVICE, this);
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {
            this.SaveCommand = new VCommand(this.Save);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void InitMessage()
        {
            ApplicationDomainEx.MessageManager.Register<ControlFieldChangedMessage>(this, this.OnControlFieldChangedMessage);
        }

        // =============================================================
        // Field & Service
        // =============================================================

        /// <summary>
        /// 控制对象服务
        /// </summary>
        private ControlObjectService controlObjectService = new ControlObjectService();

        // =============================================================
        // Property
        // =============================================================

        #region ControlObject -- 控制对象

        private ControlObjectModel controlObject;
        /// <summary>
        /// 控制对象
        /// </summary>
        public ControlObjectModel ControlObject
        {
            get { return controlObject; }
            set { controlObject = value; this.RaisePropertyChanged(nameof(ControlObject)); }
        }

        #endregion

        #region ControlField -- 控制字段

        private ControlFieldNodeModel controlField;
        /// <summary>
        /// 控制字段
        /// </summary>
        public ControlFieldNodeModel ControlField
        {
            get { return controlField; }
            set { controlField = value; this.RaisePropertyChanged(nameof(ControlField)); }
        }

        #endregion

        // =============================================================
        // Command
        // =============================================================

        #region SaveCommand -- 保存命令

        /// <summary>
        /// 保存命令
        /// </summary>
        public VCommand SaveCommand { get; set; }

        /// <summary>
        /// 保存
        /// </summary>
        private void Save()
        {
            // 当前没有打开的页或模板 || 当前没有打开的项目
            if (ApplicationDomainEx.CurrentPage == null || ApplicationDomainEx.ProjectDbContext == null)
            {
                DXMessageBox.Show("保存失败！");
                return;
            }

            // 字段树服务不可用
            IFieldTreeService service = ApplicationDomainEx.ServiceManager.GetService<IFieldTreeService>(ViewServiceKeys.FIELD_TREE_SERVICE);
            if (service == null)
            {
                DXMessageBox.Show("保存失败！");
                return;
            }

            // 当前没有控制对象
            ControlObjectModel obj = service.GetControlObject();
            if (obj == null)
            {
                return;
            }

            // 保存
            this.controlObjectService.SaveControlFields(ApplicationDomainEx.CurrentPage, obj.AllFiledNodes);

            DXMessageBox.Show("保存成功！");
        }

        #endregion

        // =============================================================
        // Message
        // =============================================================

        /// <summary>
        /// 控制字段切换消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnControlFieldChangedMessage(ControlFieldChangedMessage msg)
        {
            this.ControlObject = msg.ControlObject;
            this.ControlField = msg.ControlField;

            if (msg.ControlObject == null || msg.ControlField == null)
            {
                this.SelectedNavigationConfig = null;
                return;
            }

            NavigationConfig config = this.NavigationConfigs.FirstOrDefault(p => p.Key == msg.ControlField.Type.ToString());

            if (config != null && config.View != null && config.View.TryGetTarget(out object target))
            {
                FrameworkElement view = target as FrameworkElement;
                EditPanelModelBase model = view?.DataContext as EditPanelModelBase;

                if (model != null)
                {
                    model.FieldEditMode = FieldEditMode.Normal;
                    model.Update(this.ControlObject, this.ControlField);
                }
            }

            this.SelectedNavigationConfig = config;
        }

        // =============================================================
        // Public Function
        // =============================================================

        // =============================================================
        // Protected Function
        // =============================================================

        /// <summary>
        /// 视图创建后触发
        /// </summary>
        /// <param name="config">导航配置</param>
        /// <param name="obj">视图</param>
        protected override void OnViewCreated(NavigationConfig config, object obj)
        {
            FrameworkElement view = obj as FrameworkElement;
            if (view == null)
                return;

            EditPanelModelBase model = view.DataContext as EditPanelModelBase;
            if (model == null)
                return;

            model.FieldEditMode = FieldEditMode.Normal;
            model.Update(this.ControlObject, this.ControlField);
        }
    }
}
