﻿using DevExpress.Xpf.Core;
using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using VIZ.Framework.Core;
using VIZ.Package.Domain;
using VIZ.Package.Service;
using VIZ.Package.Storage;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 页分组视图模型
    /// </summary>
    public class PageGroupViewModel : ViewModelBase, IPageGroupService
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(PageGroupViewModel));

        public PageGroupViewModel()
        {
            ApplicationDomainEx.ServiceManager.AddService(ViewServiceKeys.PAGE_GROUP_SERVICE, this);

            // 初始化命令
            this.InitCommand();

            // 初始化消息
            this.InitMessage();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {
            this.TabAddGroupCommand = new VCommand<TabControlTabAddingEventArgs>(this.TabAddGroup);
            this.AddGroupCommand = new VCommand(this.AddGroup);
            this.RenameGroupCommand = new VCommand(this.RenameGroup);
            this.DeleteGroupCommand = new VCommand(this.DeleteGroup);
            this.DeleteItemCommand = new VCommand(this.DeleteItem);
            this.OpenPageCommand = new VCommand(this.OpenPage);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void InitMessage()
        {
            ApplicationDomainEx.MessageManager.Register<ProjectOpenMessage>(this, this.OnProjectOpenMessage);
            ApplicationDomainEx.MessageManager.Register<ProjectCloseMessage>(this, this.OnProjectCloseMessage);
            ApplicationDomainEx.MessageManager.Register<ProjectSaveMessage>(this, this.OnProjectSaveMessage);

        }

        // ======================================================================================
        // Controller & Service
        // ======================================================================================

        /// <summary>
        /// 页服务
        /// </summary>
        private PageService pageService = new PageService();

        /// <summary>
        /// GH服务
        /// </summary>
        private GHService ghService = new GHService();

        // ======================================================================================
        // Property
        // ======================================================================================

        #region IsEnabled -- 是否可用

        private bool isEnabled;
        /// <summary>
        /// 是否可用
        /// </summary>
        public bool IsEnabled
        {
            get { return isEnabled; }
            set { isEnabled = value; this.RaisePropertyChanged(nameof(IsEnabled)); }
        }

        #endregion

        #region IsLoading -- 是否正在加载

        private bool isLoading;
        /// <summary>
        /// 是否正在加载
        /// </summary>
        public bool IsLoading
        {
            get { return isLoading; }
            set { isLoading = value; this.RaisePropertyChanged(nameof(IsLoading)); }
        }

        #endregion

        #region PageGroupModels -- 页分组集合

        private ObservableCollection<PageGroupModel> pageGroupModels;
        /// <summary>
        /// 页分组集合
        /// </summary>
        public ObservableCollection<PageGroupModel> PageGroupModels
        {
            get { return pageGroupModels; }
            set { pageGroupModels = value; this.RaisePropertyChanged(nameof(PageGroupModels)); }
        }

        #endregion

        #region SelectedPageGroupModel -- 当前选中的页分组

        private PageGroupModel selectedPageGroupModel;
        /// <summary>
        /// 当前选中的页分组
        /// </summary>
        public PageGroupModel SelectedPageGroupModel
        {
            get { return selectedPageGroupModel; }
            set
            {
                selectedPageGroupModel = value;
                this.RaisePropertyChanged(nameof(SelectedPageGroupModel));

                this.SelectedPageModel = null;
            }
        }

        #endregion

        #region SelectedPageModel -- 当前选中的页

        private PageModel selectedPageModel;
        /// <summary>
        /// 当前选中的页
        /// </summary>
        public PageModel SelectedPageModel
        {
            get { return selectedPageModel; }
            set { selectedPageModel = value; this.RaisePropertyChanged(nameof(SelectedPageModel)); }
        }

        #endregion

        // ======================================================================================
        // Command
        // ======================================================================================

        #region TabAddGroupCommand -- 页签添加分组命令

        /// <summary>
        /// 添加节目单命令
        /// </summary>
        public VCommand<TabControlTabAddingEventArgs> TabAddGroupCommand { get; set; }

        /// <summary>
        /// 添加节目单
        /// </summary>
        private void TabAddGroup(TabControlTabAddingEventArgs e)
        {
            PageGroupModel model = new PageGroupModel();
            model.GroupID = Guid.NewGuid();
            model.GroupName = "新建分组";

            e.Item = model;
        }

        #endregion

        #region AddGroupCommand -- 添加分组命令

        /// <summary>
        /// 添加分组命令
        /// </summary>
        public VCommand AddGroupCommand { get; set; }

        /// <summary>
        /// 添加分组
        /// </summary>
        private void AddGroup()
        {
            PageGroupModel model = new PageGroupModel();
            model.GroupID = Guid.NewGuid();
            model.GroupName = "新建分组";

            this.PageGroupModels.Add(model);
        }

        #endregion

        #region RenameGroup -- 重命名分组命令

        /// <summary>
        /// 重命名分组命令
        /// </summary>
        public VCommand RenameGroupCommand { get; set; }

        /// <summary>
        /// 重命名分组
        /// </summary>
        private void RenameGroup()
        {
            PageGroupRenameWindow window = new PageGroupRenameWindow();
            PageGroupRenameViewModel vm = window.DataContext as PageGroupRenameViewModel;
            vm.PageGroupName = this.SelectedPageGroupModel.GroupName;
            if (window.ShowDialog() != true)
                return;

            this.SelectedPageGroupModel.GroupName = vm.PageGroupName;
        }

        #endregion

        #region DeleteGroupCommand -- 删除分组命令

        /// <summary>
        /// 删除分组命令
        /// </summary>
        public VCommand DeleteGroupCommand { get; set; }

        /// <summary>
        /// 删除分组
        /// </summary>
        private void DeleteGroup()
        {
            if (this.SelectedPageGroupModel == null)
                return;

            if (DXMessageBox.Show($"是否删除分组【{this.SelectedPageGroupModel.GroupName}】?", "提示", MessageBoxButton.YesNo) != MessageBoxResult.Yes)
                return;

            this.PageGroupModels.Remove(this.SelectedPageGroupModel);
        }

        #endregion

        #region DeleteItemCommand -- 刪除項命令

        /// <summary>
        /// 删除项命令
        /// </summary>
        public VCommand DeleteItemCommand { get; set; }

        /// <summary>
        /// 删除项
        /// </summary>
        private void DeleteItem()
        {
            if (this.SelectedPageGroupModel == null || this.SelectedPageModel == null)
                return;

            if (DXMessageBox.Show($"是否删除项【{this.SelectedPageModel.Scene}】", "提示", MessageBoxButton.YesNo) != MessageBoxResult.Yes)
                return;

            this.SelectedPageGroupModel.Pages.Remove(this.SelectedPageModel);
        }

        #endregion

        #region OpenPageCommand -- 打开页命令

        /// <summary>
        /// 打开场景页命令
        /// </summary>
        public VCommand OpenPageCommand { get; set; }

        /// <summary>
        /// 打开场景页
        /// </summary>
        private void OpenPage()
        {
            if (this.SelectedPageModel == null)
                return;

            if (ApplicationDomainEx.CurrentPage == this.SelectedPageModel)
                return;

            ApplicationDomainEx.CurrentPage = this.SelectedPageModel;

            PageOpenMessage msg = new PageOpenMessage();
            msg.Page = this.SelectedPageModel;
            ApplicationDomainEx.MessageManager.Send(msg);
        }

        #endregion

        // ======================================================================================
        // Message
        // ======================================================================================

        /// <summary>
        /// 项目打开消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProjectOpenMessage(ProjectOpenMessage msg)
        {
            this.IsEnabled = true;
            this.PageGroupModels = null;
            this.SelectedPageGroupModel = null;
            this.SelectedPageModel = null;
            this.IsLoading = true;

            ThreadHelper.SafeRun(() =>
            {
                // 页分组
                var pageGroups = this.pageService.LoadPageGroups().ToObservableCollection();

                // 更新界面

                WPFHelper.Invoke(() =>
                {
                    this.PageGroupModels = pageGroups;
                    this.SelectedPageGroupModel = pageGroups.FirstOrDefault();

                    this.IsLoading = false;
                });

                if (this.SelectedPageGroupModel == null)
                    return;

                // 获取场景模板缩略图
                foreach (var model in this.SelectedPageGroupModel.Pages)
                {
                    var bitmap = ghService.GetImage(model.ThumbnailUri);

                    WPFHelper.BeginInvoke(() =>
                    {
                        model.ThumbnailBitmap = bitmap;
                    });
                }
            });
        }

        /// <summary>
        /// 项目关闭消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProjectCloseMessage(ProjectCloseMessage msg)
        {
            this.IsEnabled = false;
            this.IsLoading = false;
            this.PageGroupModels = null;
            this.SelectedPageGroupModel = null;
            this.SelectedPageModel = null;
        }

        /// <summary>
        /// 项目保存消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProjectSaveMessage(ProjectSaveMessage msg)
        {
            this.pageService.SavePageGroups(this.PageGroupModels);
        }

        // ======================================================================================
        // Public Function
        // ======================================================================================

        /// <summary>
        /// 添加页
        /// </summary>
        /// <param name="template">模板</param>
        public void AddPage(PageTemplateModel template)
        {
            if (this.SelectedPageGroupModel == null)
                return;

            PageAddWindow window = new PageAddWindow();
            PageAddViewModel vm = window.DataContext as PageAddViewModel;
            vm.Scene = template.Scene;
            vm.Remark = template.Remark;
            vm.Layer = template.Layer;

            if (window.ShowDialog() != true)
                return;

            PageModel page = new PageModel();
            page.PageID = Guid.NewGuid();
            page.PageGroupID = this.SelectedPageGroupModel.GroupID;
            page.Scene = template.Scene;
            page.ScenePath = template.ScenePath;
            page.ThumbnailUri = template.ThumbnailUri;
            page.Remark = vm.Remark;
            page.Layer = vm.Layer;
            page.EngineType = template.EngineType;
            page.PageType = template.PageType;
            page.TemplateID = template.TemplateID;
            page.ThumbnailBitmap = template.ThumbnailBitmap;
            page.PluginID = vm.SelectedTemplatePlugin?.ID;
            page.PluginName = vm.SelectedTemplatePlugin?.Name;

            this.SelectedPageGroupModel.Pages.Add(page);

            if (page.ThumbnailBitmap == null)
            {
                ThreadHelper.SafeRun(() =>
                {
                    var bmp = this.ghService.GetImage(page.ThumbnailUri);

                    WPFHelper.BeginInvoke(() =>
                    {
                        page.ThumbnailBitmap = bmp;
                    });
                });
            }
        }
    }
}
