﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Package.Domain;
using VIZ.Package.Storage;

namespace VIZ.Package.Service
{
    /// <summary>
    /// Viz命令服务
    /// </summary>
    public class VizCommandService
    {
        /// <summary>
        /// 设置场景
        /// </summary>
        /// <param name="conn">连接</param>
        /// <param name="scene">场景</param>
        /// <param name="layer">图层</param>
        public void SetObject(ConnModel conn, string scene, VizLayer layer)
        {
            if (conn == null)
                throw new ArgumentNullException(nameof(conn));

            // 需要等待结果
            conn.EndpointManager.Request($"RENDERER*{layer} SET_OBJECT SCENE*{scene}");
        }

        /// <summary>
        /// 重新加载场景
        /// </summary>
        /// <param name="conn">连接</param>
        /// <param name="scene">场景</param>
        public void Reload(ConnModel conn, string scene)
        {
            conn.EndpointManager.Send($"SCENE**{scene} RELOAD");
        }

        /// <summary>
        /// 播放
        /// </summary>
        /// <param name="conn">连接</param>
        /// <param name="scene">场景</param>
        /// <param name="layer">图层</param>
        public void Start(ConnModel conn, string scene, VizLayer layer)
        {
            if (conn == null)
                throw new ArgumentNullException(nameof(conn));

            if (string.IsNullOrWhiteSpace(scene))
            {
                conn.EndpointManager.Send($"RENDERER*{layer}*STAGE START");
            }
            else
            {
                conn.EndpointManager.Send($"RENDERER*{layer}*STAGE*DIRECTOR*{scene} START");
            }
        }

        /// <summary>
        /// 继续
        /// </summary>
        /// <param name="conn">连接</param>
        /// <param name="scene">场景</param>
        /// <param name="layer">图层</param>
        public void Continue(ConnModel conn, string scene, VizLayer layer)
        {
            if (conn == null)
                throw new ArgumentNullException(nameof(conn));

            if (string.IsNullOrWhiteSpace(scene))
            {
                conn.EndpointManager.Send($"RENDERER*{layer}*STAGE CONTINUE");
            }
            else
            {
                conn.EndpointManager.Send($"RENDERER*{layer}*STAGE*DIRECTOR*{scene} CONTINUE");
            }
        }

        /// <summary>
        /// 停止
        /// </summary>
        /// <param name="conn">连接</param>
        /// <param name="scene">场景</param>
        /// <param name="layer">图层</param>
        public void Stop(ConnModel conn, string scene, VizLayer layer)
        {
            if (conn == null)
                throw new ArgumentNullException(nameof(conn));

            if (string.IsNullOrWhiteSpace(scene))
            {
                conn.EndpointManager.Send($"RENDERER*{layer}*STAGE STOP");
            }
            else
            {
                conn.EndpointManager.Send($"RENDERER*{layer}*STAGE*DIRECTOR*{scene} STOP");
            }
        }

        /// <summary>
        /// 显示安全框
        /// </summary>
        /// <param name="conn">连接</param>
        /// <param name="show">是否显示</param>
        public void ShowTS(ConnModel conn, bool show)
        {
            if (conn == null)
                throw new ArgumentNullException(nameof(conn));

            VizConfigEntity config = ApplicationDomainEx.VizConfig;

            if (config.EngineFullType == EngineFullType.VIZ_Eng3)
            {
                conn.EndpointManager.Send($"RENDERER SET_TITLE_AREA {(show ? 1 : 0)}");
            }
            else if (config.EngineFullType == EngineFullType.VIZ_Eng4)
            {
                conn.EndpointManager.Send($"{ApplicationDomainEx.VizPreviewRenderer} SET_TITLE_AREA {(show ? 1 : 0)}");
            }
        }

        /// <summary>
        /// 显示安区域
        /// </summary>
        /// <param name="conn">连接</param>
        /// <param name="show">是否显示</param>
        public void ShowSA(ConnModel conn, bool show)
        {
            if (conn == null)
                throw new ArgumentNullException(nameof(conn));

            VizConfigEntity config = ApplicationDomainEx.VizConfig;

            if (config.EngineFullType == EngineFullType.VIZ_Eng3)
            {
                conn.EndpointManager.Send($"RENDERER SET_SAFE_AREA {(show ? 1 : 0)}");
            }
            else if (config.EngineFullType == EngineFullType.VIZ_Eng4)
            {
                conn.EndpointManager.Send($"{ApplicationDomainEx.VizPreviewRenderer} SET_SAFE_AREA {(show ? 1 : 0)}");
            }
        }

        /// <summary>
        /// 显示边界框
        /// </summary>
        /// <param name="conn">连接</param>
        /// <param name="show">是否显示</param>
        public void ShowBB(ConnModel conn, bool show)
        {
            if (conn == null)
                throw new ArgumentNullException(nameof(conn));

            VizConfigEntity config = ApplicationDomainEx.VizConfig;

            if (config.EngineFullType == EngineFullType.VIZ_Eng3)
            {
                conn.EndpointManager.Send($"RENDERER SHOW_BOUNDING_BOX {(show ? 1 : 0)}");
            }
            else if (config.EngineFullType == EngineFullType.VIZ_Eng4)
            {
                conn.EndpointManager.Send($"{ApplicationDomainEx.VizPreviewRenderer} SHOW_BOUNDING_BOX {(show ? 1 : 0)}");
            }
        }

        /// <summary>
        /// 显示RGB
        /// </summary>
        /// <param name="conn">连接</param>
        public void ShowRGB(ConnModel conn)
        {
            if (conn == null)
                throw new ArgumentNullException(nameof(conn));

            VizConfigEntity config = ApplicationDomainEx.VizConfig;

            if (config.EngineFullType == EngineFullType.VIZ_Eng3)
            {
                conn.EndpointManager.Send($"RENDERER SET_KEY 0,RENDERER*KEY_INTERNAL*ACTIVE SET 0");
            }
            else if (config.EngineFullType == EngineFullType.VIZ_Eng4)
            {
                conn.EndpointManager.Send($"{ApplicationDomainEx.VizPreviewRenderer} SET_KEY 0,RENDERER*KEY_INTERNAL*ACTIVE SET 0");
            }
        }

        /// <summary>
        /// 显示键
        /// </summary>
        /// <param name="conn">连接</param>
        public void ShowKey(ConnModel conn)
        {
            if (conn == null)
                throw new ArgumentNullException(nameof(conn));

            VizConfigEntity config = ApplicationDomainEx.VizConfig;

            if (config.EngineFullType == EngineFullType.VIZ_Eng3)
            {
                conn.EndpointManager.Send($"RENDERER SET_KEY 1,RENDERER*KEY_INTERNAL*ACTIVE SET 0");
            }
            else if (config.EngineFullType == EngineFullType.VIZ_Eng4)
            {
                conn.EndpointManager.Send($"{ApplicationDomainEx.VizPreviewRenderer} SET_KEY 1,RENDERER*KEY_INTERNAL*ACTIVE SET 0");
            }
        }

        /// <summary>
        /// 显示预览键
        /// </summary>
        /// <param name="conn">连接</param>
        public void ShowKeyPreview(ConnModel conn)
        {
            if (conn == null)
                throw new ArgumentNullException(nameof(conn));

            VizConfigEntity config = ApplicationDomainEx.VizConfig;

            if (config.EngineFullType == EngineFullType.VIZ_Eng3)
            {
                conn.EndpointManager.Send($"RENDERER SET_KEY 0,RENDERER*KEY_INTERNAL*IMAGE RESET,RENDERER*KEY_INTERNAL*ACTIVE SET 1");
            }
            else if (config.EngineFullType == EngineFullType.VIZ_Eng4)
            {
                conn.EndpointManager.Send($"{ApplicationDomainEx.VizPreviewRenderer} SET_KEY 0,RENDERER*KEY_INTERNAL*IMAGE RESET,RENDERER*KEY_INTERNAL*ACTIVE SET 1");
            }
        }

        /// <summary>
        /// 上版子
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="scene">场景</param>
        /// <param name="layer">层</param>s
        public void TakeIn(ConnModel conn, string scene, VizLayer layer)
        {
            conn.EndpointManager.Send("RENDERER*UPDATE SET 0");
            conn.EndpointManager.Send($"RENDERER*{layer} SET_OBJECT {scene}");
            conn.EndpointManager.Send("RENDERER*UPDATE SET 1");
            conn.EndpointManager.Send($"RENDERER*{layer}*STAGE START");
        }

        /// <summary>
        /// 继续版子
        /// </summary>
        /// <param name="conn">连接</param>
        /// <param name="layer">层</param>
        public void TakeContinue(ConnModel conn, VizLayer layer)
        {
            conn.EndpointManager.Send($"RENDERER*UPDATE SET 1");
            conn.EndpointManager.Send($"RENDERER*{layer}*STAGE CONTINUE");
        }

        /// <summary>
        /// 下版子
        /// </summary>
        /// <param name="conn">连接</param>
        /// <param name="layer">层</param>s
        public void TakeOut(ConnModel conn, VizLayer layer)
        {
            conn.EndpointManager.Send($"RENDERER*UPDATE SET 1");
            conn.EndpointManager.Send($"RENDERER*{layer} SET_OBJECT");
        }
    }
}
