﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using VIZ.Framework.Common;
using VIZ.Framework.Core;
using VIZ.Package.Domain;
using VIZ.Package.Storage;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 字段编辑视图模型
    /// </summary>
    public class FieldEditViewModel : ViewModelBase
    {
        public FieldEditViewModel()
        {
            // 初始化导航配置
            this.InitNavigationConfigs();

            // 初始化消息
            this.InitMessage();
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void InitMessage()
        {
            ApplicationDomainEx.MessageManager.Register<ControlFieldChangedMessage>(this, this.OnControlFieldChangedMessage);
        }

        /// <summary>
        /// 初始化导航配置
        /// </summary>
        private void InitNavigationConfigs()
        {
            // 文本编辑器
            this.NavigationConfigs.Add(new NavigationConfig
            {
                Key = VizControlFieldType.text.ToString(),
                ViewType = typeof(TextEditPanel),
                ViewCreated = this.OnViewCreated
            });

            // 图片选择
            this.NavigationConfigs.Add(new NavigationConfig
            {
                Key = VizControlFieldType.image.ToString(),
                ViewType = typeof(ImageEditPanel),
                ViewCreated = this.OnViewCreated
            });

            // 列表编辑
            this.NavigationConfigs.Add(new NavigationConfig
            {
                Key = VizControlFieldType.list.ToString(),
                ViewType = typeof(ListEditPanel),
                ViewCreated = this.OnViewCreated
            });
        }

        // =============================================================
        // Property
        // =============================================================

        #region NavigationConfigs -- 导航配置

        private List<NavigationConfig> navigationConfigs = new List<NavigationConfig>();
        /// <summary>
        /// 导航配置
        /// </summary>
        public List<NavigationConfig> NavigationConfigs
        {
            get { return navigationConfigs; }
            set { navigationConfigs = value; this.RaisePropertyChanged(nameof(NavigationConfigs)); }
        }

        #endregion

        #region SelectedNavigationConfig -- 当前选中的导航配置

        private NavigationConfig selectedNavigationConfig;
        /// <summary>
        /// 当前选中的导航配置
        /// </summary>
        public NavigationConfig SelectedNavigationConfig
        {
            get { return selectedNavigationConfig; }
            set { selectedNavigationConfig = value; this.RaisePropertyChanged(nameof(SelectedNavigationConfig)); }
        }

        #endregion

        #region ControlObject -- 控制对象

        private ControlObjectModel controlObject;
        /// <summary>
        /// 控制对象
        /// </summary>
        public ControlObjectModel ControlObject
        {
            get { return controlObject; }
            set { controlObject = value; this.RaisePropertyChanged(nameof(ControlObject)); }
        }

        #endregion

        #region ControlField -- 控制字段

        private ControlFieldNodeModel controlField;
        /// <summary>
        /// 控制字段
        /// </summary>
        public ControlFieldNodeModel ControlField
        {
            get { return controlField; }
            set { controlField = value; this.RaisePropertyChanged(nameof(ControlField)); }
        }

        #endregion

        // =============================================================
        // Message
        // =============================================================

        /// <summary>
        /// 控制字段切换消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnControlFieldChangedMessage(ControlFieldChangedMessage msg)
        {
            this.ControlObject = msg.ControlObject;
            this.ControlField = msg.ControlField;

            if (msg.ControlObject == null || msg.ControlField == null)
            {
                this.SelectedNavigationConfig = null;
                return;
            }

            NavigationConfig config = this.NavigationConfigs.FirstOrDefault(p => p.Key == msg.ControlField.Type.ToString());

            if (config != null && config.View != null && config.View.TryGetTarget(out object target))
            {
                FrameworkElement view = target as FrameworkElement;
                EditPanelModelBase model = view?.DataContext as EditPanelModelBase;

                if (model != null)
                {
                    model.FieldEditMode = FieldEditMode.Normal;
                    model.Update(this.ControlObject, this.ControlField);
                }
            }

            this.SelectedNavigationConfig = config;
        }

        // =============================================================
        // Private Function
        // =============================================================

        /// <summary>
        /// 视图创建后触发
        /// </summary>
        /// <param name="config">导航配置</param>
        /// <param name="obj">视图</param>
        private void OnViewCreated(NavigationConfig config, object obj)
        {
            FrameworkElement view = obj as FrameworkElement;
            if (view == null)
                return;

            EditPanelModelBase model = view.DataContext as EditPanelModelBase;
            if (model == null)
                return;

            model.FieldEditMode = FieldEditMode.Normal;
            model.Update(this.ControlObject, this.ControlField);
        }
    }
}
