﻿using DevExpress.Xpf.Core;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using VIZ.Framework.Core;
using VIZ.Package.Domain;
using VIZ.Package.Storage;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 主视图顶部视图模型
    /// </summary>
    public class MainTopViewModel : ViewModelBase
    {
        public MainTopViewModel()
        {
            // 初始化命令
            this.InitCommand();

            // 初始化消息
            this.InitMessage();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {
            this.CreateProjectCommand = new VCommand(this.CreateProject);
            this.OpenProjectCommand = new VCommand(this.OpenProject);
            this.SaveProjectCommand = new VCommand(this.SaveProject, this.CanSaveProject);
            this.CloseProjectCommand = new VCommand(this.CloseProject, this.CanCloseProject);

            this.SettingCommand = new VCommand(this.Setting);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void InitMessage()
        {
            ApplicationDomainEx.MessageManager.Register<VizPreviewReadyMessage>(this, this.OnVizPreviewReadyMessage);
        }

        // =====================================================================
        // Property
        // =====================================================================

        #region ProjectName -- 项目名

        private string projectName;
        /// <summary>
        /// 项目名
        /// </summary>
        public string ProjectName
        {
            get { return projectName; }
            set { projectName = value; this.RaisePropertyChanged(nameof(ProjectName)); }
        }

        #endregion

        #region IsVizPreviewReady -- Viz预览是否准备完毕

        private bool isVizPreviewReady;
        /// <summary>
        /// Viz预览是否准备完毕
        /// </summary>
        public bool IsVizPreviewReady
        {
            get { return isVizPreviewReady; }
            set { isVizPreviewReady = value; this.RaisePropertyChanged(nameof(IsVizPreviewReady)); }
        }

        #endregion

        // =====================================================================
        // Command
        // =====================================================================

        #region CreateProjectCommand -- 创建项目命令

        /// <summary>
        /// 创建项目命令
        /// </summary>
        public VCommand CreateProjectCommand { get; set; }

        /// <summary>
        /// 创建项目
        /// </summary>
        private void CreateProject()
        {
            // 保存之前的项目
            if (ApplicationDomainEx.ProjectDbContext != null)
            {
                var result = DXMessageBox.Show("是否保存当前项目", "提示", System.Windows.MessageBoxButton.YesNo);

                if (result == System.Windows.MessageBoxResult.Yes)
                {
                    this.SaveProject();
                }
            }

            // 创建项目文件
            SaveFileDialog sfd = new SaveFileDialog();
            sfd.Filter = "viz_pkg files(*.viz_pkg)|*.viz_pkg";
            if (sfd.ShowDialog() != DialogResult.OK)
                return;

            this.ProjectName = System.IO.Path.GetFileNameWithoutExtension(sfd.FileName);
            ApplicationDomainEx.ProjectDbContext = new ProjectDbContext(sfd.FileName);

            // 为项目添加默认页分组
            PageGroupEntity defaultGroup = new PageGroupEntity();
            defaultGroup.GroupID = Guid.NewGuid();
            defaultGroup.GroupName = "默认分组";
            ApplicationDomainEx.ProjectDbContext.PageGroup.Insert(defaultGroup);

            // 发送项目打开消息
            ProjectOpenMessage msg = new ProjectOpenMessage();
            ApplicationDomainEx.MessageManager.Send(msg);

            this.UpdateCommandStatus();
        }

        #endregion

        #region OpenProjectCommand -- 打开项目命令

        /// <summary>
        /// 打开项目命令
        /// </summary>
        public VCommand OpenProjectCommand { get; set; }

        /// <summary>
        /// 打开项目
        /// </summary>
        private void OpenProject()
        {
            if (ApplicationDomainEx.ProjectDbContext != null)
            {
                var result = DXMessageBox.Show("是否保存当前项目", "提示", System.Windows.MessageBoxButton.YesNo);

                if (result == System.Windows.MessageBoxResult.Yes)
                {
                    this.SaveProject();
                }
            }

            OpenFileDialog ofd = new OpenFileDialog();
            ofd.Filter = "viz_pkg files(*.viz_pkg)|*.viz_pkg";
            ofd.Multiselect = false;
            if (ofd.ShowDialog() != DialogResult.OK)
                return;

            this.ProjectName = System.IO.Path.GetFileNameWithoutExtension(ofd.FileName);
            ApplicationDomainEx.ProjectDbContext = new ProjectDbContext(ofd.FileName);

            ProjectOpenMessage msg = new ProjectOpenMessage();
            ApplicationDomainEx.MessageManager.Send(msg);

            this.UpdateCommandStatus();
        }

        #endregion

        #region SaveProjectCommand -- 保存项目命令

        /// <summary>
        /// 保存项目命令
        /// </summary>
        public VCommand SaveProjectCommand { get; set; }

        /// <summary>
        /// 是否可以执行保存项目命令
        /// </summary>
        /// <returns>是否可以执行保存项目命令</returns>
        private bool CanSaveProject()
        {
            return ApplicationDomainEx.ProjectDbContext != null;
        }

        /// <summary>
        /// 保存项目
        /// </summary>
        private void SaveProject()
        {
            ProjectSaveMessage msg = new ProjectSaveMessage();
            ApplicationDomainEx.MessageManager.Send(msg);
        }

        #endregion

        #region CloseProjectCommand -- 关闭项目命令

        /// <summary>
        /// 关闭项目命令
        /// </summary>
        public VCommand CloseProjectCommand { get; set; }

        /// <summary>
        /// 是否可以执行关闭项目
        /// </summary>
        /// <returns>是否可以执行关闭项目</returns>
        private bool CanCloseProject()
        {
            return ApplicationDomainEx.ProjectDbContext != null;
        }

        /// <summary>
        /// 关闭项目
        /// </summary>
        private void CloseProject()
        {
            if (ApplicationDomainEx.ProjectDbContext != null)
            {
                var result = DXMessageBox.Show("是否保存当前项目", "提示", System.Windows.MessageBoxButton.YesNo);

                if (result == System.Windows.MessageBoxResult.Yes)
                {
                    this.SaveProject();
                }
            }

            this.ProjectName = null;
            ApplicationDomainEx.ProjectDbContext = null;

            ProjectCloseMessage msg = new ProjectCloseMessage();
            ApplicationDomainEx.MessageManager.Send(msg);

            this.UpdateCommandStatus();
        }

        #endregion

        #region SettingCommand -- 设置命令

        /// <summary>
        /// 设置命令
        /// </summary>
        public VCommand SettingCommand { get; set; }

        /// <summary>
        /// 设置
        /// </summary>
        private void Setting()
        {

        }

        #endregion

        // =====================================================================
        // Message
        // =====================================================================

        /// <summary>
        /// Viz预览准备完毕消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnVizPreviewReadyMessage(VizPreviewReadyMessage msg)
        {
            this.IsVizPreviewReady = true;
        }

        // =====================================================================
        // Public Function
        // =====================================================================

        /// <summary>
        /// 更新命令状态
        /// </summary>
        public void UpdateCommandStatus()
        {
            this.SaveProjectCommand.RaiseCanExecute();
            this.CloseProjectCommand.RaiseCanExecute();
        }
    }
}
