﻿using log4net;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.Package.Domain;
using VIZ.Package.Plugin;
using VIZ.Package.Service;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 控制视图模型
    /// </summary>
    public class ControlViewModel : ViewModelBase
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(ControlViewModel));

        public ControlViewModel()
        {
            // 初始化命令
            this.InitCommand();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {
            this.TakeCommand = new VCommand(this.Take);
            this.ContinueCommand = new VCommand(this.Continue);
            this.TakeOutCommand = new VCommand(this.TakeOut);
            this.UpdateCommand = new VCommand(this.Update);
        }

        // ==============================================================
        // Service & Controller
        // ==============================================================

        /// <summary>
        /// Viz引擎命令服务
        /// </summary>
        private VizCommandService vizCommandService = new VizCommandService();

        /// <summary>
        /// Viz引擎控制对象服务
        /// </summary>
        private VizCommandControlObjectService vizCommandControlObjectService = new VizCommandControlObjectService();

        // ==============================================================
        // Property
        // ==============================================================

        // ==============================================================
        // Command
        // ==============================================================

        #region TakeCommand -- 上版子命令

        /// <summary>
        /// 上版命令
        /// </summary>
        public VCommand TakeCommand { get; set; }

        /// <summary>
        /// 上板
        /// </summary>
        private void Take()
        {
            this.Execute(
                controlObjectAction: (obj, conn) =>
                {
                    this.vizCommandService.TakeIn(conn, ApplicationDomainEx.CurrentPage.ScenePath, ApplicationDomainEx.CurrentPage.Layer);
                },
                pluginAction: (view, conns) =>
                {
                    view.TakIn(conns);
                });
        }

        #endregion

        #region ContinueCommand -- 继续命令

        /// <summary>
        /// 继续命令
        /// </summary>
        public VCommand ContinueCommand { get; set; }

        /// <summary>
        /// 继续
        /// </summary>
        private void Continue()
        {
            this.Execute(
                controlObjectAction: (obj, conn) =>
                {
                    this.vizCommandService.Continue(conn, ApplicationDomainEx.CurrentPage.ScenePath, ApplicationDomainEx.CurrentPage.Layer);
                },
                pluginAction: (view, conns) =>
                {
                    view.TakeContinue(conns);
                });
        }

        #endregion

        #region TakeOutCommand -- 下版命令

        /// <summary>
        /// 下版命令
        /// </summary>
        public VCommand TakeOutCommand { get; set; }

        /// <summary>
        /// 下版
        /// </summary>
        private void TakeOut()
        {
            this.Execute(
                controlObjectAction: (obj, conn) =>
                {
                    this.vizCommandService.TakeOut(conn, ApplicationDomainEx.CurrentPage.Layer);
                },
                pluginAction: (view, conns) =>
                {
                    view.TakeOut(conns);
                });
        }

        #endregion

        #region UpdateCommand -- 更新命令

        /// <summary>
        /// 更新命令
        /// </summary>
        public VCommand UpdateCommand { get; set; }

        /// <summary>
        /// 更新
        /// </summary>
        private void Update()
        {
            this.Execute(
                controlObjectAction: (obj, conn) =>
                {
                    this.vizCommandControlObjectService.SetControlObject(conn, obj);
                },
                pluginAction: (view, conns) =>
                {
                    view.TakeUpdate(conns);
                });
        }

        #endregion

        // ==============================================================
        // Private Function
        // ==============================================================

        /// <summary>
        /// 执行
        /// </summary>
        /// <param name="controlObjectAction">控制对象行为</param>
        /// <param name="pluginAction">插件行为</param>
        private void Execute(Action<ControlObjectModel, ConnModel> controlObjectAction, Action<IPluginView, IList<ConnModel>> pluginAction)
        {
            // 是否拥有打开的页或模板页
            if (ApplicationDomainEx.CurrentPage == null)
                return;

            // 是否有启用的连接分组
            ConnGroupModel group = ApplicationDomainEx.ConnGroups.FirstOrDefault(p => p.IsEnabled);
            if (group == null)
                return;

            IFieldTreeService fieldTreeService = ApplicationDomainEx.ServiceManager.GetService<IFieldTreeService>(ViewServiceKeys.FIELD_TREE_SERVICE);
            ControlObjectModel controlObject = fieldTreeService?.GetControlObject();

            // Step 1. 处理控制对象
            if (controlObject != null)
            {
                foreach (var item in group.Items)
                {
                    if (!item.IsEnabled || !item.IsConnected)
                        continue;

                    try
                    {
                        controlObjectAction(controlObject, item);
                    }
                    catch (Exception ex)
                    {
                        log.Error(ex);
                    }
                }
            }

            // Step 2. 执行插件脚本
            if (!(ApplicationDomainEx.CurrentPage is PageModel page))
                return;

            if (string.IsNullOrWhiteSpace(page.PluginID))
                return;

            IPluginService pluginService = ApplicationDomainEx.ServiceManager.GetService<IPluginService>(ViewServiceKeys.PLUGIN_SERVICE);
            if (pluginService == null)
                return;

            IPluginView view = pluginService.GetCurrentPluginView() as IPluginView;
            if (view == null)
                return;

            try
            {
                pluginAction(view, group.Items.Where(p => p.IsEnabled && p.IsConnected).ToList());
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }
        }
    }
}