﻿using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.Package.Connection;
using VIZ.Package.Domain;
using VIZ.Package.Service;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 主连接视图模型
    /// </summary>
    public class MainConnViewModel : ViewModelBase
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(MainConnViewModel));

        public MainConnViewModel()
        {
            // 初始化命令
            this.InitCommnad();

            // 初始化消息
            this.InitMessage();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommnad()
        {
            this.ItemClickCommand = new VCommand<ConnModel>(this.ItemClick);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void InitMessage()
        {
            ApplicationDomainEx.MessageManager.Register<ConnChangedMessage>(this, this.OnConnChangedMessage);
            ApplicationDomainEx.MessageManager.Register<VizPreviewReadyMessage>(this, this.OnVizPreviewReadyMessage);
        }

        // ==========================================================================
        // Field & Service
        // ==========================================================================

        /// <summary>
        /// 连接服务
        /// </summary>
        private ConnService connService = new ConnService();

        // ==========================================================================
        // Property
        // ==========================================================================

        #region ItemsSrouce -- 数据源

        private ObservableCollection<ConnModel> itemsSrouce = new ObservableCollection<ConnModel>();
        /// <summary>
        /// 数据源
        /// </summary>
        public ObservableCollection<ConnModel> ItemsSrouce
        {
            get { return itemsSrouce; }
            set { itemsSrouce = value; this.RaisePropertyChanged(nameof(ItemsSrouce)); }
        }

        #endregion

        // ==========================================================================
        // Command
        // ==========================================================================

        #region ItemClickCommand -- 项点击命令

        /// <summary>
        /// 项点击命令
        /// </summary>
        public VCommand<ConnModel> ItemClickCommand { get; set; }

        /// <summary>
        /// 项点击命令
        /// </summary>
        /// <param name="model">连接模型</param>
        private void ItemClick(ConnModel model)
        {
            model.IsLoading = true;

            ThreadHelper.SafeRun(action: () =>
            {
                if (model.IsConnected)
                {
                    model.EndpointManager.Disconnect();
                }
                else
                {
                    model.EndpointManager.Connect();
                }
            }, final: () =>
            {
                model.IsLoading = false;
            });
        }

        #endregion

        // ==========================================================================
        // Message
        // ==========================================================================

        /// <summary>
        /// 连接改变消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnConnChangedMessage(ConnChangedMessage msg)
        {
            if (msg.OldGroup != null)
            {
                foreach (ConnModel item in msg.OldGroup.Items)
                {
                    item.Dispose();
                }
            }

            this.InitAndConnnectGroups(msg.NewGroup);
        }

        /// <summary>
        /// Viz引擎初始化完成消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnVizPreviewReadyMessage(VizPreviewReadyMessage msg)
        {
            List<ConnGroupModel> groups = this.connService.LoadGroups();
            ApplicationDomainEx.ConnGroups = groups;

            ConnGroupModel group = groups.FirstOrDefault(p => p.IsEnabled == true);

            this.InitAndConnnectGroups(group);
        }

        /// <summary>
        /// 初始化 & 连接
        /// </summary>
        /// <param name="group">连接分组</param>
        private void InitAndConnnectGroups(ConnGroupModel group)
        {
            this.ItemsSrouce.Clear();

            // 预览连接
            this.ItemsSrouce.Add(ApplicationDomainEx.PreviewConn);

            if (group == null)
                return;

            foreach (ConnModel item in group.Items)
            {
                if (!item.IsEnabled)
                    continue;

                item.InitEndpointManager(new VizEndpointManager(item.ConnID.ToString(), item.IP, item.Port));

                this.ItemsSrouce.Add(item);

                // 启动连接
                ThreadHelper.SafeRun(() =>
                {
                    try
                    {
                        item.EndpointManager.Connect();
                    }
                    catch (Exception ex)
                    {
                        log.Error(ex);
                    }
                });
            }
        }
    }
}
