﻿using DevExpress.Xpf.Printing;
using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Common;
using VIZ.Framework.Core;
using VIZ.Package.Domain;
using VIZ.Package.Plugin;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 插件视图模型
    /// </summary>
    public class PluginViewModel : ViewModelBase, IPluginService
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(PluginViewModel));

        public PluginViewModel()
        {
            // 初始化消息
            this.InitMessage();

            // 注册服务
            ApplicationDomainEx.ServiceManager.AddService(ViewServiceKeys.PLUGIN_SERVICE, this);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void InitMessage()
        {
            ApplicationDomainEx.MessageManager.Register<ProjectCloseMessage>(this, this.OnProjectCloseMessage);
            ApplicationDomainEx.MessageManager.Register<PageInitedMessage>(this, this.OnPageInitedMessage);
        }

        // ========================================================================
        // Property
        // ========================================================================

        #region SelectedValue -- 当前选择项

        private NavigationConfig selectedValue;
        /// <summary>
        /// 当前选择项
        /// </summary>
        public NavigationConfig SelectedValue
        {
            get { return selectedValue; }
            set { selectedValue = value; this.RaisePropertyChanged(nameof(SelectedValue)); }
        }

        #endregion

        #region ItemsSource -- 数据源

        private ObservableCollection<NavigationConfig> itemsSource = new ObservableCollection<NavigationConfig>();
        /// <summary>
        /// 数据源
        /// </summary>
        public ObservableCollection<NavigationConfig> ItemsSource
        {
            get { return itemsSource; }
            set { itemsSource = value; this.RaisePropertyChanged(nameof(ItemsSource)); }
        }

        #endregion

        // ========================================================================
        // Command
        // ========================================================================

        // ========================================================================
        // Message
        // ========================================================================

        /// <summary>
        /// 项目关闭消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProjectCloseMessage(ProjectCloseMessage msg)
        {
            this.SelectedValue = null;

            foreach (NavigationConfig config in this.ItemsSource)
            {
                if (config.View == null || !config.View.TryGetTarget(out object target))
                    continue;

                if (!(target is IPluginView view))
                    continue;

                try
                {
                    view.Dispose();
                }
                catch (Exception ex)
                {
                    log.Error(ex);
                }
            }
        }

        /// <summary>
        /// 页初始化完成消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnPageInitedMessage(PageInitedMessage msg)
        {
            // 页模板没有插件
            if (msg.Page == null || msg.Page is PageTemplateModel)
            {
                this.SelectedValue = null;
                return;
            }

            // 页拥有插件
            PageModel page = msg.Page as PageModel;

            NavigationConfig config = this.ItemsSource.FirstOrDefault(p => p.Key == page.PageID.ToString());
            if (config != null)
            {
                this.SelectedValue = config;
                return;
            }

            PluginInfo plugin = ApplicationDomainEx.PluginInfos.FirstOrDefault(p => p.ID == page.PluginID);
            if (plugin == null)
                return;

            config = new NavigationConfig();
            config.Key = page.PageID.ToString();
            config.ViewType = plugin.ViewType;
            this.ItemsSource.Add(config);

            this.SelectedValue = config;
        }

        // ========================================================================
        // Public Function
        // ========================================================================

        /// <summary>
        /// 获取当前插件视图
        /// </summary>
        /// <returns>当前插件视图</returns>
        public object GetCurrentPluginView()
        {
            if (this.SelectedValue == null)
                return null;

            this.SelectedValue.View.TryGetTarget(out object target);

            return target;
        }
    }
}
