﻿using DevExpress.Xpf.Core;
using log4net;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using VIZ.Framework.Core;
using VIZ.Package.Connection;
using VIZ.Package.Domain;
using VIZ.Package.Service;
using VIZ.Package.Storage;

namespace VIZ.Package.Module
{
    /// <summary>
    /// Viz预览视图模型
    /// </summary>
    public class VizPreviewViewModel : ViewModelBase
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(VizPreviewViewModel));

        public VizPreviewViewModel()
        {
            // 初始化控制器
            this.initController();

            // 初始化命令
            this.initCommand();

            // 初始化消息
            this.initMessage();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void initCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
            this.WindowHostSizeChangedCommand = new VCommand<System.Windows.SizeChangedEventArgs>(this.WindowHostSizeChanged);
            this.PlayCommand = new VCommand(this.Play);
            this.ContinueCommand = new VCommand(this.Continue);
            this.StopCommand = new VCommand(this.Stop);
            this.ShowTSCommand = new VCommand(this.ShowTS);
            this.HideTSCommand = new VCommand(this.HideTS);
            this.ShowSACommand = new VCommand(this.ShowSA);
            this.HideSACommand = new VCommand(this.HideSA);
            this.ShowBBCommand = new VCommand(this.ShowBB);
            this.HideBBCommand = new VCommand(this.HideBB);
            this.ShowRGBCommand = new VCommand(this.ShowRGB);
            this.ShowKeyCommand = new VCommand(this.ShowKey);
            this.ShowKeyPreviewCommand = new VCommand(this.ShowKeyPreview);
        }

        /// <summary>
        /// 初始化控制器
        /// </summary>
        private void initController()
        {
            this.vizPreviewController = new VizPreviewController(this);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void initMessage()
        {
            ApplicationDomainEx.MessageManager.Register<PageOpenMessage>(this, this.OnPageOpenMessage);
            ApplicationDomainEx.MessageManager.Register<ApplicationCloseMessage>(this, this.OnApplicationCloseMessage);
        }

        // ================================================================================
        // Property
        // ================================================================================

        #region IsEngineReady -- 引擎是否准备完毕

        private bool isEngineReady;
        /// <summary>
        /// 引擎是否准备完毕
        /// </summary>
        public bool IsEngineReady
        {
            get { return isEngineReady; }
            set { isEngineReady = value; this.RaisePropertyChanged(nameof(IsEngineReady)); }
        }

        #endregion

        // ================================================================================
        // Controller & Service
        // ================================================================================

        /// <summary>
        /// Viz预览控制器
        /// </summary>
        private VizPreviewController vizPreviewController;

        /// <summary>
        /// VIZ命令服务
        /// </summary>
        private VizCommandService vizCommandService = new VizCommandService();

        // ================================================================================
        // Command
        // ================================================================================

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {
            if (this.IsAlreadyLoaded)
                return;

            VizPreviewView view = this.GetView<VizPreviewView>();
            if (view == null)
                return;

            ThreadHelper.SafeRun(() =>
            {
                // 等待能正确获取到界面宽度时再嵌入VIZ引擎
                double width = 0;
                while (width == 0)
                {
                    WPFHelper.Invoke(() =>
                    {
                        width = view.ActualWidth;
                    });

                    System.Threading.Thread.Sleep(500);
                }

                // 启动引擎
                this.StartVizEngine();
            });

            this.IsAlreadyLoaded = true;
        }

        #endregion

        #region WindowHostSizeChangedCommand -- 宿主容器大小改变命令

        /// <summary>
        /// 宿主容器大小改变命令
        /// </summary>
        public VCommand<System.Windows.SizeChangedEventArgs> WindowHostSizeChangedCommand { get; set; }

        /// <summary>
        /// 宿主容器大小改变
        /// </summary>
        private void WindowHostSizeChanged(System.Windows.SizeChangedEventArgs e)
        {
            VizConfigEntity config = ApplicationDomainEx.VizConfig;
            ConnModel conn = ApplicationDomainEx.PreviewConn;

            if (conn == null || !conn.IsConnected)
                return;

            var dpi = WPFHelper.GetDpiByGraphics();

            int width = (int)(e.NewSize.Width * (dpi.X / 96d));
            int height = (int)(e.NewSize.Height * (dpi.Y / 96d));

            ImageHelper.AdjustSize(width, height, 1920, 1080, out width, out height);

            if (config.EngineFullType == EngineFullType.VIZ_Eng3)
            {
                conn.EndpointManager.Send($"RENDERER WINDOW_RESIZE {width} {height}");
            }
            else
            {
                conn.EndpointManager.Send($"{ApplicationDomainEx.VizPreviewRenderer} WINDOW_RESIZE {width} {height}");
            }
        }

        #endregion

        #region PlayCommand -- 播放命令

        /// <summary>
        /// 播放命令
        /// </summary>
        public VCommand PlayCommand { get; set; }

        /// <summary>
        /// 播放
        /// </summary>
        private void Play()
        {
            this.Execute(
                action: (conn) =>
                {
                    this.vizCommandService.Start(conn, null, VizLayer.MAIN_LAYER);
                },
                pluginAction: (view, conn) =>
                {
                    view.PreviewIn(conn);
                });
        }

        #endregion

        #region ContinueCommand -- 继续命令

        /// <summary>
        /// 继续命令
        /// </summary>
        public VCommand ContinueCommand { get; set; }

        /// <summary>
        /// 继续
        /// </summary>
        private void Continue()
        {
            this.Execute(
                action: (conn) =>
                {
                    this.vizCommandService.Continue(conn, null, VizLayer.MAIN_LAYER);
                },
                pluginAction: (view, conn) =>
                {
                    view.PreviewContinue(conn);
                });
        }

        #endregion

        #region StopCommand -- 停止命令

        /// <summary>
        /// 停止命令
        /// </summary>
        public VCommand StopCommand { get; set; }

        /// <summary>
        /// 停止
        /// </summary>
        private void Stop()
        {
            this.Execute(
                action: (conn) =>
                {
                    this.vizCommandService.Stop(conn, null, VizLayer.MAIN_LAYER);
                },
                pluginAction: (view, conn) =>
                {
                    view.PreviewOut(conn);
                });
        }

        #endregion

        // --------------------------------
        // 安全框

        #region ShowTSCommand -- 显示安全框命令

        /// <summary>
        /// 显示安全框命令
        /// </summary>
        public VCommand ShowTSCommand { get; set; }

        /// <summary>
        /// 显示安全框
        /// </summary>
        private void ShowTS()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.CurrentPage == null)
                return;

            this.vizCommandService.ShowTS(ApplicationDomainEx.PreviewConn, true);
        }

        #endregion

        #region HideTSCommand -- 隐藏安全框命令

        /// <summary>
        /// 隐藏安全框命令
        /// </summary>
        public VCommand HideTSCommand { get; set; }

        /// <summary>
        /// 隐藏安全框
        /// </summary>
        private void HideTS()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.CurrentPage == null)
                return;

            this.vizCommandService.ShowTS(ApplicationDomainEx.PreviewConn, false);
        }

        #endregion

        // --------------------------------
        // 安全区域

        #region ShowSACommand -- 显示安全区域命令

        /// <summary>
        /// 显示安全区域命令
        /// </summary>
        public VCommand ShowSACommand { get; set; }

        /// <summary>
        /// 显示安全区域
        /// </summary>
        private void ShowSA()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.CurrentPage == null)
                return;

            this.vizCommandService.ShowSA(ApplicationDomainEx.PreviewConn, true);
        }

        #endregion

        #region HideSACommand -- 隐藏安全区域命令

        /// <summary>
        /// 隐藏安全区域命令
        /// </summary>
        public VCommand HideSACommand { get; set; }

        /// <summary>
        /// 隐藏安全区域
        /// </summary>
        private void HideSA()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.CurrentPage == null)
                return;

            this.vizCommandService.ShowSA(ApplicationDomainEx.PreviewConn, false);
        }

        #endregion

        // --------------------------------
        // 边界框

        #region ShowBBCommand -- 显示边界框命令

        /// <summary>
        /// 显示边界框命令
        /// </summary>
        public VCommand ShowBBCommand { get; set; }

        /// <summary>
        /// 显示边界框
        /// </summary>
        private void ShowBB()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.CurrentPage == null)
                return;

            this.vizCommandService.ShowBB(ApplicationDomainEx.PreviewConn, true);
        }

        #endregion

        #region HideBBCommand -- 隐藏边界框命令

        /// <summary>
        /// 隐藏边界框命令
        /// </summary>
        public VCommand HideBBCommand { get; set; }

        /// <summary>
        /// 隐藏边界框
        /// </summary>
        private void HideBB()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.CurrentPage == null)
                return;

            this.vizCommandService.ShowBB(ApplicationDomainEx.PreviewConn, false);
        }

        #endregion

        // --------------------------------
        // 其他

        #region ShowRGBCommand -- 显示RGB命令

        /// <summary>
        /// 显示RGB命令
        /// </summary>
        public VCommand ShowRGBCommand { get; set; }

        /// <summary>
        /// 显示RGB
        /// </summary>
        private void ShowRGB()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.CurrentPage == null)
                return;

            this.vizCommandService.ShowRGB(ApplicationDomainEx.PreviewConn);
        }

        #endregion

        #region ShowKeyCommand -- 显示键命令

        /// <summary>
        /// 显示键命令
        /// </summary>
        public VCommand ShowKeyCommand { get; set; }

        /// <summary>
        /// 显示键
        /// </summary>
        private void ShowKey()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.CurrentPage == null)
                return;

            this.vizCommandService.ShowKey(ApplicationDomainEx.PreviewConn);
        }

        #endregion

        #region ShowKeyPreviewCommand -- 显示预览键命令

        /// <summary>
        /// 显示预览键命令
        /// </summary>
        public VCommand ShowKeyPreviewCommand { get; set; }

        /// <summary>
        /// 显示预览键
        /// </summary>
        private void ShowKeyPreview()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.CurrentPage == null)
                return;

            this.vizCommandService.ShowKeyPreview(ApplicationDomainEx.PreviewConn);
        }

        #endregion

        // ================================================================================
        // Message
        // ================================================================================

        /// <summary>
        /// 页打开消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnPageOpenMessage(PageOpenMessage msg)
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.CurrentPage == null)
                return;

            this.vizCommandService.SetObject(ApplicationDomainEx.PreviewConn, msg.Page.ScenePath, VizLayer.MAIN_LAYER);

            // 发送节目单项初始化完成消息
            WPFHelper.BeginInvoke(() =>
            {
                PageInitedMessage message = new PageInitedMessage();
                message.Page = msg.Page;

                ApplicationDomainEx.MessageManager.Send(message);
            });
        }

        /// <summary>
        /// 应用程序关闭消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnApplicationCloseMessage(ApplicationCloseMessage msg)
        {
            if (ApplicationDomainEx.VizPreviewProcess == null || ApplicationDomainEx.VizPreviewProcess.HasExited)
                return;

            try
            {
                ApplicationDomainEx.VizPreviewProcess.Kill();
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }
        }

        // ================================================================================
        // Public Function
        // ================================================================================

        /// <summary>
        /// 启动VIZ引擎
        /// </summary>
        public void StartVizEngine()
        {
            VizPreviewView view = this.GetView<VizPreviewView>();
            if (view == null)
                return;

            VizConfigEntity vizConfig = ApplicationDomainEx.VizConfig;
            ConnModel conn = ApplicationDomainEx.PreviewConn;
            if (conn != null)
            {
                conn.Dispose();
            }

            conn = new ConnModel();
            conn.IP = vizConfig.VIZ_IP;
            conn.Port = vizConfig.VIZ_Port;
            conn.Remark = "预览";
            conn.InitEndpointManager(new VizEndpointManager("local", vizConfig.VIZ_IP, vizConfig.VIZ_Port));

            ApplicationDomainEx.PreviewConn = conn;

            string path = null;

            switch (vizConfig.EngineFullType)
            {
                case EngineFullType.VIZ_Eng3: path = vizConfig.VIZ_Eng3Path; break;
                case EngineFullType.VIZ_Eng4: path = vizConfig.VIZ_Eng4Path; break;
            }

            this.vizPreviewController.StartVizEngine(view, path, conn);
        }

        // ================================================================================
        // Private Function
        // ================================================================================

        /// <summary>
        /// 执行
        /// </summary>
        /// <param name="action">播放行为</param>
        /// <param name="pluginAction">插件行为</param>
        private void Execute(Action<ConnModel> action, Action<IPluginView, ConnModel> pluginAction)
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.CurrentPage == null)
                return;

            // 预览连接为准备好
            if (ApplicationDomainEx.PreviewConn == null || !ApplicationDomainEx.PreviewConn.IsConnected)
                return;

            // Step 1. 播放行为
            try
            {
                action(ApplicationDomainEx.PreviewConn);
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }

            // Step 2. 插件行为
            if (!(ApplicationDomainEx.CurrentPage is PageModel page))
                return;

            if (string.IsNullOrWhiteSpace(page.PluginID))
                return;

            IPluginService pluginService = ApplicationDomainEx.ServiceManager.GetService<IPluginService>(ViewServiceKeys.PLUGIN_SERVICE);
            if (pluginService == null)
                return;

            IPluginView view = pluginService.GetCurrentPluginView() as IPluginView;
            if (view == null)
                return;

            try
            {
                pluginAction(view, ApplicationDomainEx.PreviewConn);
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }
        }
    }
}
