﻿using System;
using System.Collections.Generic;
using System.Dynamic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.Package.Domain;
using VIZ.Package.Service;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 富本编辑面板模型
    /// </summary>
    public class RichTextEditPanelModel : EditPanelModelBase
    {
        /// <summary>
        /// 富文本左侧
        /// </summary>
        public const string RICH_TEXT_LEFT = "<fo:wrapper xmlns:fo=\"http://www.w3.org/1999/XSL/Format\"><![CDATA[";

        /// <summary>
        /// 富文本右侧
        /// </summary>
        public const string RICH_TEXT_RIGHT = "]]></fo:wrapper>";

        // ============================================================
        // Property
        // ============================================================

        #region Text -- 文本

        private string text;
        /// <summary>
        /// 文本
        /// </summary>
        public string Text
        {
            get { return text; }
            set
            {
                text = value;
                this.RaisePropertyChanged(nameof(Text));
                this.OnTextChanged();
            }
        }

        #endregion

        // ============================================================
        // Message
        // ============================================================

        // ============================================================
        // Public Function
        // ============================================================

        /// <summary>
        /// 更新
        /// </summary>
        /// <param name="controlObject">控制对象</param>
        /// <param name="controlField">控制字段</param>
        public override void Update(ControlObjectModel controlObject, ControlFieldNodeModel controlField)
        {
            base.Update(controlObject, controlField);

            this.IsSendToPreview = false;
            this.Text = this.GetTextFromRichText(controlField?.Value);
            this.IsSendToPreview = true;
        }

        /// <summary>
        /// 更新动态数据
        /// </summary>
        /// <param name="listCellEdit">列单元格编辑器</param>
        /// <param name="columnDefinition">列定义</param>
        /// <param name="rowHandle">行号</param>
        /// <param name="row">行数据</param>
        public override void UpdateDynamic(ListCellEditBase listCellEdit, GridColumnDefinition columnDefinition, int rowHandle, ExpandoObject row)
        {
            base.UpdateDynamic(listCellEdit, columnDefinition, rowHandle, row);

            IDictionary<string, object> dic = row as IDictionary<string, object>;

            this.IsSendToPreview = false;
            this.Text = this.GetTextFromRichText(dic?[columnDefinition.FieldName]?.ToString());
            this.IsSendToPreview = true;
        }

        /// <summary>
        /// 获取字段值
        /// </summary>
        /// <returns>字段值</returns>
        public override string GetFieldValue()
        {
            return this.Text;
        }

        // ============================================================
        // Private Function
        // ============================================================

        /// <summary>
        /// 文本值改变时触发
        /// </summary>
        private void OnTextChanged()
        {
            // 不需要向预览发送值
            if (!this.IsSendToPreview)
                return;

            // 没有预览连接
            if (ApplicationDomainEx.PreviewConn == null)
                return;

            // 没有控制对象或控制字段
            if (this.ControlObject == null || this.ControlField == null)
                return;

            // 正常模式编辑
            if (this.FieldEditMode == FieldEditMode.Normal)
            {
                string richText = $"{RICH_TEXT_LEFT}{this.Text}{RICH_TEXT_RIGHT}";
                this.ControlField.Value = richText;

                this.VizCommandControlObjectService.SetControlObjectValue(
                    ApplicationDomainEx.PreviewConn,
                    this.ControlObject.TreeNodePath,
                    this.ControlField.FieldIdentifier,
                    this.ControlField.Value);

                return;
            }

            // 没有列信息或行数据
            if (this.ColumnDefinition == null || this.Row == null)
                return;

            // 动态模式编辑
            if (this.FieldEditMode == FieldEditMode.Dynamic)
            {
                IDictionary<string, object> dic = this.Row as IDictionary<string, object>;
                string richText = $"{RICH_TEXT_LEFT}{this.Text}{RICH_TEXT_RIGHT}";
                dic[this.ColumnDefinition.FieldName] = richText;

                this.ListCellEdit.UpdateEditValue(this.ColumnDefinition, this.RowHandle, this.Row);

                return;
            }
        }

        /// <summary>
        /// 从富文本中获取文本内容
        /// </summary>
        /// <param name="richText">富文本</param>
        /// <returns>文本内容</returns>
        private string GetTextFromRichText(string richText)
        {
            if (string.IsNullOrWhiteSpace(richText))
                return richText;

            string str = richText.Replace(RICH_TEXT_LEFT, string.Empty);
            str = str.Replace(RICH_TEXT_RIGHT, string.Empty);

            return str;
        }
    }
}
