﻿using DevExpress.Mvvm.Xpf;
using DevExpress.Xpf.Core;
using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Configuration;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Net;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Forms;
using System.Windows.Shapes;
using VIZ.Framework.Core;
using VIZ.Package.Domain;


namespace VIZ.Package.Module
{
    public class MediaResourcePanelViewModel : ViewModelBase, IMediaResourceFileSupport
    {
        /// <summary>
        /// 日志
        /// </summary>
        private static readonly ILog log = LogManager.GetLogger(typeof(MediaResourceViewModel));

        public MediaResourcePanelViewModel()
        {
            // 初始化控制器
            this.initController();

            // 初始化命令
            this.initCommand();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void initCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);


            this.RefreshFolderCommand = new VCommand(this.RefreshFolder);
            this.RefreshFileCommand = new VCommand(this.RefreshFile);

            this.CreateFloderCommand = new VCommand(this.CreateFloder);
            this.DeleteFloderCommand = new VCommand(this.DeleteFloder);
            this.CreateFileCommand = new VCommand(this.CreateFile);
            this.DeleteFileCommand = new VCommand(this.DeleteFile);
            this.MoveFileCommand = new VCommand(this.MoveFile);
            this.MoveFloderCommand = new VCommand(this.MoveFileFloder);
            this.ExportFileCommand = new VCommand(this.ExprotFile);
            this.ExportFolderFilesCommand = new VCommand(this.ExportFolderFiles);
            this.FilesListCommand = new VCommand(this.FilesList);


            this.ShowImageListCommand = new VCommand(this.ShowImageList);
            this.SortFileNameCommand = new VCommand(this.SortFileName);
            this.SortDateCommand = new VCommand(this.SortDate);

            this.FolderExpandCommand = new VCommand<DevExpress.Xpf.Grid.TreeList.NodeDoubleClickEventArgs>(this.FolderExpand);
            this.FileFilterCommand = new VCommand<ResourceFileType>(this.FileFilter);
            this.FileRowFilterCommand = new DevExpress.Mvvm.DelegateCommand<RowFilterArgs>(this.FileRowFilter);
            this.FileDoubleClickCommand = new VCommand(this.FileDoubleClick);
            this.EditImageCommand = new VCommand(EditImage);
            //this.FileContextMenuOpendCommand = new VCommand(this.FileContextMenuOpend);
            //this.AddProgramTemplateCommand = new VCommand(this.AddProgramTemplate, this.CanAddProgramTemplate);
            // this.CopyFilePathCommand = new VCommand(this.CopyFilePath, this.CanCopyFilePath);

        }

        /// <summary>
        /// 初始化控制器
        /// </summary>
        private void initController()
        {
            this.mediaResourceFileController = new MediaResourceFileController(this);
        }

        // ==================================================================================
        // Property
        // ==================================================================================

        #region FolderModels -- 文件夹目录集合

        private ObservableCollection<MHResourceFolderModel> folderModels;
        /// <summary>
        /// 文件夹目录集合
        /// </summary>
        public ObservableCollection<MHResourceFolderModel> FolderModels
        {
            get { return folderModels; }
            set { folderModels = value; this.RaisePropertyChanged(nameof(FolderModels)); }
        }

        #endregion

        #region SelectedFolderModel -- 当前选中的文件夹

        private MHResourceFolderModel selectedFolderModel;
        /// <summary>
        /// 当前选中的文件夹
        /// </summary>
        public MHResourceFolderModel SelectedFolderModel
        {
            get { return selectedFolderModel; }
            set
            {
                selectedFolderModel = value;
                this.RaisePropertyChanged(nameof(SelectedFolderModel));
                // 更新文件模型
                this.mediaResourceFileController.UpdateFileModels(value);
            }
        }

        #endregion

        #region FileModels -- 文件集合

        private ObservableCollection<MHResourceFileModel> fileModels;
        /// <summary>
        /// 文件集合
        /// </summary>
        public ObservableCollection<MHResourceFileModel> FileModels
        {
            get { return fileModels; }
            set { fileModels = value; this.RaisePropertyChanged(nameof(FileModels)); }
        }

        #endregion

        #region SelectedFileModel -- 选中的文件模型

        private MHResourceFileModel selectedFileModel;
        /// <summary>
        /// 选中的文件模型
        /// </summary>
        public MHResourceFileModel SelectedFileModel
        {
            get { return selectedFileModel; }
            set
            {
                selectedFileModel = value;
                this.RaisePropertyChanged(nameof(SelectedFileModel));
                this.OnSelectedFileModelChanged?.Invoke(this, new MHResourceSelectedFileChangedEventArgs { File = value });
            }
        }

        #endregion

        #region IsFolderLoading -- 是否正在加载文件夹

        private bool isFolderLoading;
        /// <summary>
        /// 是否正在加载文件夹
        /// </summary>
        public bool IsFolderLoading
        {
            get { return isFolderLoading; }
            set { isFolderLoading = value; this.RaisePropertyChanged(nameof(IsFolderLoading)); }
        }

        #endregion

        #region IsFileLoading -- 是否正在加载文件

        private bool isFileLoading;
        /// <summary>
        /// 是否正在加载文件
        /// </summary>
        public bool IsFileLoading
        {
            get { return isFileLoading; }
            set { isFileLoading = value; this.RaisePropertyChanged(nameof(IsFileLoading)); }
        }

        #endregion

        #region FilterResourceFileType -- 资源文件类型

        private ResourceFileType filterResourceFileType;
        /// <summary>
        /// 资源文件类型
        /// </summary>
        public ResourceFileType FilterResourceFileType
        {
            get { return filterResourceFileType; }
            set { filterResourceFileType = value; this.RaisePropertySaveChanged(nameof(FilterResourceFileType)); }
        }

        #endregion

        // ==================================================================================
        // Service & Controller
        // ==================================================================================

        /// <summary>
        /// GH 资源服务
        /// </summary>
        //private IGHResourceService ghResourceService = new GHResourceService();

        /// <summary>
        /// VIZ资源文件控制器
        /// </summary>
        private MediaResourceFileController mediaResourceFileController;

        // ==================================================================================
        // Command
        // ==================================================================================

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {
            if (this.IsAlreadyLoaded)
                return;

            this.IsFolderLoading = true;

            // 刷新文件夹
            this.RefreshFolder();

            this.IsAlreadyLoaded = true;
        }

        #endregion


        /// <summary>
        /// 选择文件改变时触发
        /// </summary>
        public event EventHandler<MHResourceSelectedFileChangedEventArgs> OnSelectedFileModelChanged;

        /// <summary>
        /// 文件双击时触发
        /// </summary>
        public event EventHandler<MHResourceFileDoubleClickEventArgs> OnFileDoubleClick;

        #region FileDoubleClickCommand -- 文件双击命令

        /// <summary>
        /// 文件双击命令
        /// </summary>
        public VCommand FileDoubleClickCommand { get; set; }

        /// <summary>
        /// 文件双击
        /// </summary>
        private void FileDoubleClick()
        {
            this.OnFileDoubleClick?.Invoke(this, new MHResourceFileDoubleClickEventArgs { File = this.SelectedFileModel });
        }

        #endregion


        #region RefreshFolderCommand -- 刷新文件夹命令

        /// <summary>
        /// 刷新文件夹命令
        /// </summary>
        public VCommand RefreshFolderCommand { get; set; }

        /// <summary>
        /// 刷新文件夹
        /// </summary>
        private void RefreshFolder()
        {
            this.IsFolderLoading = false;

            Task.Run(async () =>
            {
                try
                {


                    List<MHResourceFolderModel> list = await this.mediaResourceFileController.GetMeiaResourceFolder();

                    if (list == null)
                    {
                        return;
                    }


                    this.FolderModels = list.ToObservableCollection();
                    this.IsFolderLoading = false;
                }
                catch (Exception ex)
                {
                    log.Error(ex);
                }
            });
        }

        #endregion

        #region RefreshFileCommand -- 刷新文件命令



        /// <summary>
        /// 刷新文件命令
        /// </summary>
        public VCommand RefreshFileCommand { get; set; }

        /// <summary>
        /// 刷新文件
        /// </summary>
        private void RefreshFile()
        {
            MHResourceFolderModel folder = this.SelectedFolderModel;

            if (folder == null)
                return;

            this.mediaResourceFileController.DisposeFileModels(folder);
            folder.IsRefreshedFiles = false;
            this.mediaResourceFileController.UpdateFileModels(this.SelectedFolderModel);
        }

        #endregion


        #region ImageList可见性
        private Visibility visibilityImage=Visibility.Visible;

        public Visibility VisibilityImage
        {
            get { return visibilityImage; }
            set { visibilityImage = value; this.RaisePropertyChanged(nameof(VisibilityImage)); }
        }

        #endregion


        #region 文件列表
        private Visibility visibilityList=Visibility.Hidden;

        public Visibility VisibilityList
        {
            get { return visibilityList; }
            set { visibilityList = value; this.RaisePropertyChanged(nameof(VisibilityList)); }
        }
        #endregion 

        #region 创建文件夹

        public VCommand CreateFloderCommand { get; set; }

        /// <summary>
        /// 创建文件夹
        /// </summary>
        private async void CreateFloder()
        {
            CreateFolderWindow floderView = new CreateFolderWindow();

            CreateFolderViewModel vm = floderView.createFolderView.DataContext as CreateFolderViewModel;

            floderView.ShowDialog();

            if (vm == null || !vm.IsEnter)
                return;

            if (this.FolderModels == null)
                return;
            string createPath = "";
            //创建根目录文件夹
            if (this.FolderModels.Count == 0)
            {
                createPath = string.Format("{0}\\{1}", this.mediaResourceFileController.RootPath, vm.FolderName);
                var result = await this.mediaResourceFileController.CreateFolder(createPath);
                if (result != null)
                {
                    if (result.status == "0")
                    {

                        this.folderModels.Add(new MHResourceFolderModel()
                        {
                            Name = vm.FolderName,
                            Path = createPath,
                            FolderType = ResourceFolderType.Folder
                        });
                    }
                    else
                    {
                        log.Error(result.message);
                    }

                }

            }
            else
            {
                createPath = string.Format("{0}\\{1}", selectedFolderModel.Path, vm.FolderName);
                var result = await this.mediaResourceFileController.CreateFolder(createPath);
                if (result != null)
                {
                    if (result.status == "0")
                    {
                        SelectedFolderModel.Children.Add(new MHResourceFolderModel()
                        {
                            Name = vm.FolderName,
                            Path = createPath,
                            FolderType = ResourceFolderType.Folder,
                            Parent = selectedFolderModel
                        });
                    }
                    else
                    {
                        log.Error(result.message);
                    }

                }
            }
        }

        #endregion

        #region 删除文件夹

        public VCommand DeleteFloderCommand { get; set; }

        private async void DeleteFloder()
        {
            if (selectedFolderModel != null)
            {
                if ((DialogResult)DXMessageBox.Show("是否此文件夹", "", MessageBoxButton.OKCancel) == DialogResult.OK)
                {
                    string path = SelectedFolderModel.Path;
                    var result = await this.mediaResourceFileController.DeleteFolder(path);

                    if (result != null)
                    {
                        if (SelectedFolderModel.Parent != null)
                        {
                            if (result.status == "0")
                            {
                                SelectedFolderModel.Parent.Children.Remove(SelectedFolderModel);
                            }
                            else
                            {
                                log.Error(result.message);
                            }
                        }
                        else
                        {
                            this.FolderModels.Remove(SelectedFolderModel);
                        }

                    }
                }
            }
        }
        #endregion

        #region 创建文件

        public VCommand CreateFileCommand { get; set; }

        /// <summary>
        /// 创建文件方法
        /// </summary>
        private async void CreateFile()
        {

            System.Windows.Forms.OpenFileDialog ofd = new System.Windows.Forms.OpenFileDialog();

            ofd.Filter = ApplicationDomainEx.MediaConfig.MediaFilter;
            ofd.Multiselect = true;
            if (ofd.ShowDialog() == System.Windows.Forms.DialogResult.OK)
            {
                string[] fileNames = ofd.FileNames;

                bool flag = false;
                foreach (var tempFileName in fileNames)
                {
                    string path = System.IO.Path.GetFullPath(tempFileName);
                    string fileName = ThumbnailHelper.GetFileFolderName(path);
                    try
                    {

                        var fileResult = await this.mediaResourceFileController.CreateFile(path, fileName);
                        if (fileResult != null && fileResult.errCode == "0")
                        {
                            flag = true;
                        }
                        else
                        {
                            flag = false;
                        }
                    }
                    catch (Exception ex)
                    {
                        log.Error(ex.Message);
                    }
                    //if (fileResult != null && fileResult.errCode == "0")
                    //{
                    //try
                    //{
                    //    //MHResourceFileModel fileModel = new MHResourceFileModel();

                    //if (ThumbnailHelper.IsImageByName(fileName))
                    //{
                    //    Image image = ThumbnailHelper.GetImage(path, 100, 200);
                    //    Bitmap btm = new Bitmap(image);
                    //    fileModel.ThumbnailBitmap = btm;
                    //    fileModel.FileType = ResourceFileType.IMAGE;
                    //    fileModel.Name = fileName;
                    //    this.FileModels.Add(fileModel);
                    //}
                    //else if (ThumbnailHelper.IsVideo(fileName))
                    //{
                    //    string vedioImage = string.Format("{0}{1}", System.Environment.CurrentDirectory, "\\Resource\\VedioImage\\Vedio.jpeg");
                    //    Image image = ThumbnailHelper.GetImage(vedioImage, 100, 200);
                    //    Bitmap btm = new Bitmap(image);
                    //    fileModel.ThumbnailBitmap = btm;
                    //    fileModel.FileType = ResourceFileType.Video;
                    //    fileModel.Name = fileName;
                    //    this.FileModels.Add(fileModel);
                    //}

                    // }
                    //catch (Exception ex)
                    //{
                    //    log.Error(ex.Message);
                    //}
                    // }
                }

                if (flag)
                {
                    if (this.SelectedFolderModel != null)
                    {
                        this.mediaResourceFileController.UpdateFileModels(this.SelectedFolderModel);
                    }

                }

            }
        }
        #endregion 创建文件

        #region 删除文件

        public VCommand DeleteFileCommand { get; set; }

        private async void DeleteFile()
        {
            if (selectedFileModel == null) return;
            if ((DialogResult)DXMessageBox.Show("是否删除此文件", "", MessageBoxButton.OKCancel) == DialogResult.OK)
            {

                var result = await this.mediaResourceFileController.DeleteFile(selectedFileModel.Path);
                if (result != null)
                {
                    if (result.status == "0")
                    {
                        this.FileModels.Remove(this.SelectedFileModel);
                    }
                    else
                    {
                        log.Error(result.message);
                    }
                }


            }
        }

        #endregion

        #region 移动文件

        public VCommand MoveFileCommand { get; set; }


        public string moveFilePath = "";
        public MHResourceFileModel moveFileModel = null;


        private void MoveFile()
        {
            if (selectedFileModel != null)
            {
                if ((DialogResult)DXMessageBox.Show("是否移动此文件", "", MessageBoxButton.OKCancel) == DialogResult.OK)
                {

                    moveFilePath = selectedFileModel.Path;
                    moveFileModel = selectedFileModel;
                }
            }

        }


        #region 查看文件列表
        public VCommand FilesListCommand { get; set; }


        private void FilesList()
        {
            VisibilityImage = Visibility.Hidden;
            VisibilityList = Visibility.Visible;
        }

        #endregion


        #region 查看图片列表
        public VCommand ShowImageListCommand { get; set; }

        private void ShowImageList()
        {
            VisibilityImage = Visibility.Visible;
            VisibilityList = Visibility.Hidden;
        }

        #endregion


        #region 文件列表按照文件名排序
        public VCommand SortFileNameCommand { get; set; }

        private void SortFileName()
        {
            this.FileModels = this.FileModels.OrderBy(a=>a.Name).ToObservableCollection();
        }

        #endregion


        #region 文件列表按照日期排序
        public VCommand SortDateCommand { get; set; }

        private void SortDate()
        {
            this.FileModels = this.FileModels.OrderBy(a=>Convert.ToDateTime(a.Date)).ToObservableCollection();
        }

        #endregion





        /// <summary>
        /// 导出文件夹的素材
        /// </summary>
        public VCommand ExportFolderFilesCommand { get; set; }


        private void ExportFolderFiles()
        {

            if (SelectedFolderModel == null || FileModels == null) return;

            try
            {
                System.Windows.Forms.FolderBrowserDialog ofd = new System.Windows.Forms.FolderBrowserDialog();
                if (ofd.ShowDialog() == System.Windows.Forms.DialogResult.OK)
                {
                    string path = ofd.SelectedPath;
                    foreach (var fileModel in FileModels)
                    {
                        string filePath = String.Format("{0}\\{1}", path, fileModel.Name);

                        if (!File.Exists(filePath))
                        {
                            fileModel.OriginThumbnailBitmap.Save(filePath);
                        }
                    }

                    DXMessageBox.Show("素材导出成功");

                }
            }

            catch (Exception ex)
            {
                DXMessageBox.Show("素材导出失败");
                log.Error(ex.Message);
            }

        }


        /// <summary>
        /// 导入文件素材
        /// </summary>


        public VCommand ExportFileCommand { get; set; }


        private void ExprotFile()
        {
            //if (SelectedFileModel == null) return;
            //var saveFileDialog1 = new SaveFileDialog
            //{
            //    Filter = "JPeg Image|*.jpg|Bitmap Image|*.bmp|Gif Image|*.gif",
            //    Title = "Save File"
            //};
            //saveFileDialog1.ShowDialog();

            //if (string.IsNullOrEmpty(saveFileDialog1.FileName)) return;

            //Bitmap bitmap = SelectedFileModel.ThumbnailBitmap;
            //bitmap.Save(saveFileDialog1.FileName);
            //using (var sw = new StreamWriter(selectedFileModel.ThumbnailBitmap))
            //{
            //    sw.Write("Writing Text ");
            //    sw.WriteLine("Here we go.");
            //    sw.WriteLine("-------------------");
            //    sw.Write("Today is is: " + DateTime.Now);
            //    sw.WriteLine("Done");
            //}


            if (SelectedFileModel == null) return;
            try
            {
                System.Windows.Forms.FolderBrowserDialog ofd = new System.Windows.Forms.FolderBrowserDialog();
                if (ofd.ShowDialog() == System.Windows.Forms.DialogResult.OK)
                {
                    string path = ofd.SelectedPath;

                    string filePath = String.Format("{0}\\{1}", path, SelectedFileModel.Name);

                    if (!File.Exists(filePath))
                    {
                        SelectedFileModel.OriginThumbnailBitmap.Save(filePath);
                    }
                    DXMessageBox.Show("素材导出成功");
                }
            }

            catch (Exception ex)
            {
                DXMessageBox.Show("素材导出失败");
                log.Error(ex.Message);
            }
        }

        public VCommand MoveFloderCommand { get; set; }

        /// <summary>
        /// 移动文件到文件夹
        /// </summary>

        private async void MoveFileFloder()
        {
            if (string.IsNullOrEmpty(moveFilePath))
            {

                // DXMessageBox.Show("请先选择移入的文件,在选择移动此文件夹","移入当前文件夹", MessageBoxButton.YesNo);
                DXMessageBox.Show("请先选择移入的文件,在选择移动此文件夹");
                // MessageBox.Show("请先选择移入的文件");
            }
            else
            {
                var result = await this.mediaResourceFileController.MoveFile(moveFilePath, selectedFolderModel.Path);

                if (result != null)
                {
                    if (result.status == "0")
                    {
                        if (moveFileModel != null)
                        {
                            this.FileModels.Add(moveFileModel);
                        }
                    }
                    else
                    {
                        log.Error(result.message);
                    }
                }


            }
        }
        #endregion

        #region

        /// <summary>
        /// 编辑图片
        /// </summary>
        public VCommand EditImageCommand { get; set; }

        private async void EditImage()
        {

            if (selectedFileModel == null || selectedFileModel.ThumbnailBitmap == null) return;

            ImageEditWindow  imageEditWindow = new ImageEditWindow();

            ImageEditViewModel vm = imageEditWindow.ImageCut.DataContext as ImageEditViewModel;

            vm.OriginImage = vm.BitmapToBitmapImage(selectedFileModel.ThumbnailBitmap);

            imageEditWindow.ShowDialog();

            if(vm.IsEnter)
            {
                string fileName = string.Format("{0}.png", GenerateStringID());

                Bitmap bp = new Bitmap(vm.CutImage);

                bool flag = false;
                try
                {

                    var fileResult = await this.mediaResourceFileController.CreateCutImageFile(bp, fileName);
                    if (fileResult != null && fileResult.errCode == "0")
                    {
                        flag = true;
                    }
                    else
                    {
                        flag = false;
                    }
                }
                catch (Exception ex)
                {
                    log.Error(ex.Message);
                }

                if (flag)
                {
                    if (this.SelectedFolderModel != null)
                    {
                        this.mediaResourceFileController.UpdateFileModels(this.SelectedFolderModel);
                    }

                }
            }

        }

        /// <summary>
        /// 随机生成GuID字符串
        /// </summary>
        /// <returns></returns>
        private string GenerateStringID()
        {
            long i = 1;
            foreach (byte b in Guid.NewGuid().ToByteArray())
            {
                i *= ((int)b + 1);
            }
            return string.Format("{0:x}", i - DateTime.Now.Ticks);
        }

        #endregion




        #region FolderExpandCommand -- 文件夹展开命令

        /// <summary>
        /// 文件夹展开命令
        /// </summary>
        public VCommand<DevExpress.Xpf.Grid.TreeList.NodeDoubleClickEventArgs> FolderExpandCommand { get; set; }

        /// <summary>
        /// 文件夹展开
        /// </summary>
        private void FolderExpand(DevExpress.Xpf.Grid.TreeList.NodeDoubleClickEventArgs e)
        {
            if (this.SelectedFolderModel == null || e.ChangedButton != System.Windows.Input.MouseButton.Left)
                return;

            this.SelectedFolderModel.IsExpand = !this.SelectedFolderModel.IsExpand;
        }

        #endregion

        #region FileFilterCommand -- 文件过滤命令

        /// <summary>
        /// 文件过滤命令
        /// </summary>
        public VCommand<ResourceFileType> FileFilterCommand { get; set; }

        /// <summary>
        /// 文件过滤
        /// </summary>
        /// <param name="type"></param>
        private void FileFilter(ResourceFileType type)
        {
            this.FilterResourceFileType = type;

            MediaResourcePanel view = this.GetView<MediaResourcePanel>();
            if (view == null)
                return;

            view.fileGrid.RefreshData();
        }

        #endregion

        #region FileContextMenuOpendCommand -- 文件右键菜单打开命令

        /// <summary>
        /// 文件右键菜单打开命令
        /// </summary>
        public VCommand FileContextMenuOpendCommand { get; set; }

        /// <summary>
        /// 文件右键菜单打开
        /// </summary>
        private void FileContextMenuOpend()
        {
            this.AddProgramTemplateCommand.RaiseCanExecute();
            this.CopyFilePathCommand.RaiseCanExecute();
        }

        #endregion

        #region AddProgramTemplateCommand -- 添加节目模板命令

        /// <summary>
        /// 添加节目模板命令
        /// </summary>
        public VCommand AddProgramTemplateCommand { get; set; }

        /// <summary>
        /// 是否可以执行添加节目模板
        /// </summary>
        /// <returns>是否可以执行添加节目模板</returns>
        private bool CanAddProgramTemplate()
        {
            return this.SelectedFileModel != null && this.SelectedFileModel.FileType == ResourceFileType.SCENE;
            //&& ApplicationDomainEx.CurrentProjectDomain != null;
        }

        /// <summary>
        /// 添加节目模板
        /// </summary>
        private void AddProgramTemplate()
        {
            if (this.SelectedFileModel == null || this.SelectedFileModel.FileType != ResourceFileType.SCENE)
                return;

            //IProgramListViewService service = ApplicationDomainEx.ServiceManager.GetService<IProgramListViewService>(ServiceKeys.PROGRAM_LIST_VIEW_SERVICE);
            //if (service == null)
            //    return;

            // service.AddSceneTemplate(this.SelectedFileModel);
        }

        #endregion

        #region FileRowFilterCommand -- 文件行筛选命令

        /// <summary>
        /// 文件行筛选命令, 该命令必须使用Dev的命令基类
        /// </summary>
        public DevExpress.Mvvm.DelegateCommand<RowFilterArgs> FileRowFilterCommand { get; set; }

        /// <summary>
        /// 文件行筛选
        /// </summary>
        /// <param name="e">赛选参数</param>
        private void FileRowFilter(RowFilterArgs e)
        {
            MHResourceFileModel fileModel = e.Item as MHResourceFileModel;
            if (fileModel == null)
            {
                e.Visible = false;
                return;
            }

            if (this.FilterResourceFileType == ResourceFileType.NONE)
            {
                e.Visible = true;
                return;
            }

            e.Visible = fileModel.FileType == this.FilterResourceFileType;
        }

        #endregion

        #region CopyFilePathCommand -- 拷贝文件路径命令

        /// <summary>
        /// 拷贝文件路径命令
        /// </summary>
        public VCommand CopyFilePathCommand { get; set; }

        /// <summary>
        /// 是否可以执行拷贝文件路径
        /// </summary>
        /// <returns>是否可以执行拷贝文件路径</returns>
        private bool CanCopyFilePath()
        {
            return this.SelectedFileModel != null;
        }

        /// <summary>
        /// 拷贝文件路径
        /// </summary>
        private void CopyFilePath()
        {
            if (this.SelectedFileModel == null)
                return;

            // Clipboard.SetText($"{this.SelectedFileModel.FileType}*{this.SelectedFileModel.Path}");
        }

        #endregion

        // ==================================================================================
        // Public Function
        // ==================================================================================

        /// <summary>
        /// 销毁
        /// </summary>
        //public override void Dispose()
        //{

        //}
    }
}
