﻿using log4net;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Package.Domain;

namespace VIZ.Package.Service
{
    /// <summary>
    /// 文件服务
    /// </summary>
    public class FileService
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(FileService));

        /// <summary>
        /// 获取指定路径下的所有文件夹
        /// </summary>
        /// <param name="path">指定路径</param>
        /// <returns>文件夹</returns>
        public List<FolderModel> GetFolders(string path)
        {
            List<FolderModel> folders = new List<FolderModel>();

            if (string.IsNullOrWhiteSpace(path) || !System.IO.Directory.Exists(path))
                return folders;

            foreach (string folder in System.IO.Directory.GetDirectories(path))
            {
                FolderModel model = new FolderModel(folder);

                folders.Add(model);
            }

            return folders;
        }

        /// <summary>
        /// 获取指定路径下的文件
        /// </summary>
        /// <param name="path">指定路径</param>
        /// <returns>文件</returns>
        public List<FileModel> GetFiles(string path)
        {
            return this.GetFiles(path, null);
        }

        /// <summary>
        /// 获取指定路径下的文件
        /// </summary>
        /// <param name="path">指定路径</param>
        /// <param name="searchPattern">匹配模式</param>
        /// <returns>文件</returns>
        public List<FileModel> GetFiles(string path, string searchPattern)
        {
            List<FileModel> files = new List<FileModel>();

            if (string.IsNullOrWhiteSpace(path) || !System.IO.Directory.Exists(path))
                return files;

            string[] fileArray = string.IsNullOrWhiteSpace(searchPattern) ?
                System.IO.Directory.GetFiles(path) :
                System.IO.Directory.GetFiles(path, searchPattern);

            foreach (string file in fileArray)
            {
                FileModel model = new FileModel(file);

                files.Add(model);
            }

            return files;
        }

        /// <summary>
        /// 创建文件夹
        /// </summary>
        /// <param name="parent">父级别文件夹</param>
        /// <param name="name">文件夹名称</param>
        /// <returns>文件夹模型</returns>
        public FolderModel CreateFolder(FolderModel parent, string name)
        {
            if (parent == null)
                throw new ArgumentNullException(nameof(parent));

            if (string.IsNullOrWhiteSpace(name))
                throw new ArgumentNullException(nameof(name));

            string path = System.IO.Path.Combine(parent.Path, name);

            if (System.IO.Directory.Exists(path))
                throw new Exception($"文件夾: {path} 已经存在.");

            Directory.CreateDirectory(path);
            FolderModel folder = new FolderModel(path);
            folder.Parent = parent;

            return folder;
        }

        /// <summary>
        /// 删除文件夹
        /// </summary>
        /// <param name="folder">文件夹</param>
        public void DeleteFolder(FolderModel folder)
        {
            if (folder == null)
                throw new ArgumentNullException(nameof(folder));

            if (!System.IO.Directory.Exists(folder.Path))
                throw new Exception($"文件夹: {folder.Path} 不存在");

            System.IO.Directory.Delete(folder.Path, true);
        }

        /// <summary>
        /// 删除文件
        /// </summary>
        /// <param name="file">文件</param>
        public void DeleteFile(FileModel file)
        {
            if (file == null)
                throw new ArgumentNullException(nameof(file));

            System.IO.File.Delete(file.Path);
        }

        /// <summary>
        /// 拷贝文件
        /// </summary>
        /// <param name="files">文件集合</param>
        /// <param name="dstFolderParent">目标文件夹集合</param>
        /// <returns>拷贝文件</returns>
        public List<FileModel> CopyFile(List<FileModel> files, FolderModel dstFolderParent)
        {
            if (files == null)
                throw new ArgumentNullException(nameof(files));

            if (dstFolderParent == null)
                throw new ArgumentNullException(nameof(dstFolderParent));

            List<FileModel> list = new List<FileModel>();

            if (files.Count == 0)
                return list;

            foreach (FileModel file in files)
            {
                string dstPath = System.IO.Path.Combine(dstFolderParent.Path, file.Name);
                System.IO.File.Copy(file.Path, dstPath);

                FileModel model = new FileModel(dstPath);
                list.Add(model);
            }

            return list;
        }
    }
}
