﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.Package.Domain;
using VIZ.Package.Service;
using VIZ.Package.Storage;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 页命令控制器
    /// </summary>
    public class PageCommandController
    {
        /// <summary>
        /// 操作日志服务
        /// </summary>
        private RecordLogService recordLogService = new RecordLogService();

        /// <summary>
        /// 延时
        /// </summary>
        /// <remarks>
        /// 示例: delay 1000 
        /// 描述：延时 1000 毫秒
        /// </remarks>
        public const string CMD_DELAY = "delay";

        /// <summary>
        /// 发送
        /// </summary>
        /// <remarks>
        /// 示例: send xxx
        /// 描述: 向viz引擎发送xxx命令
        /// </remarks>
        public const string CMD_SEND = "send";

        /// <summary>
        /// 记录日志
        /// </summary>
        /// <remarks>
        /// 示例：log xxx
        /// 描述：记录命令日志
        /// </remarks>
        public const string CMD_LOG = "log";

        /// <summary>
        /// 开始执行命令
        /// </summary>
        /// <param name="pageBase">页</param>
        /// <param name="conn">连接</param>
        /// <param name="command">命令</param>
        public void BeginExecute(PageModelBase pageBase, ConnModel conn, string command)
        {
            if (pageBase == null || string.IsNullOrWhiteSpace(command))
                return;

            ThreadHelper.SafeRun(() => this.Execute(pageBase, conn, command));
        }

        /// <summary>
        /// 执行
        /// </summary>
        /// <param name="pageBase">页</param>
        /// <param name="conn">连接</param>
        /// <param name="command">命令</param>
        private void Execute(PageModelBase pageBase, ConnModel conn, string command)
        {
            string[] lines = command.Split(new string[] { "\r", "\n" }, StringSplitOptions.RemoveEmptyEntries);

            foreach (string line in lines)
            {
                string str = line.Trim();
                int index = str.IndexOf(' ');

                string cmd = str.Substring(0, index);
                string args = str.Substring(index, str.Length - index);

                cmd = cmd.Trim().ToLower();
                args = args.Trim();

                this.ExecuteLine(pageBase, conn, cmd, args);
            }
        }

        /// <summary>
        /// 执行
        /// </summary>
        /// <param name="pageBase">页</param>
        /// <param name="conn">连接</param>
        /// <param name="cmd">命令</param>
        /// <param name="args">参数</param>
        private void ExecuteLine(PageModelBase pageBase, ConnModel conn, string cmd, string args)
        {
            switch (cmd)
            {
                case CMD_DELAY: this.ExecuteLine_Delay(pageBase, conn, cmd, args); break;
                case CMD_SEND: this.ExecuteLine_Send(pageBase, conn, cmd, args); break;
                case CMD_LOG: this.ExecuteLine_Log(pageBase, conn, cmd, args); break;
            }
        }

        /// <summary>
        /// 执行 -- 延时
        /// </summary>
        /// <param name="pageBase">页</param>
        /// <param name="conn">连接</param>
        /// <param name="cmd">命令</param>
        /// <param name="args">参数</param>
        private void ExecuteLine_Delay(PageModelBase pageBase, ConnModel conn, string cmd, string args)
        {
            int.TryParse(args, out int delay);

            if (delay <= 0)
                return;

            System.Threading.Thread.Sleep(delay);
        }

        /// <summary>
        /// 执行 -- 向Viz发送消息
        /// </summary>
        /// <param name="pageBase">页</param>
        /// <param name="conn">连接</param>
        /// <param name="cmd">命令</param>
        /// <param name="args">参数</param>
        private void ExecuteLine_Send(PageModelBase pageBase, ConnModel conn, string cmd, string args)
        {
            if (conn == null || !conn.IsEnabled || !conn.IsConnected)
                return;

            conn.EndpointManager.Send(args);
        }

        /// <summary>
        /// 执行 -- 记录命令日志
        /// </summary>
        /// <param name="pageBase">页</param>
        /// <param name="conn">连接</param>
        /// <param name="cmd">命令</param>
        /// <param name="args">参数</param>
        private void ExecuteLine_Log(PageModelBase pageBase, ConnModel conn, string cmd, string args)
        {
            // 记录操作日志
            this.recordLogService.AppendLog(ApplicationConstants.APPLICATION_GROUP_NAME, RecordLogOperate.System, RecordLogTrigger.System, RecordLogConstants.SYSTEM_PAGE_CMD_LOG, args);
        }
    }
}
