﻿using DevExpress.Xpf.Core;
using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Controls.Primitives;
using VIZ.Framework.Core;
using VIZ.Package.Domain;
using VIZ.Package.Service;
using VIZ.Package.Storage;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 页模板视图模型
    /// </summary>
    public class PageTemplateViewModel : ViewModelBase, IPageTemplateService
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(PageTemplateViewModel));

        public PageTemplateViewModel()
        {
            // 初始化命令
            this.InitCommand();

            // 初始化消息
            this.InitMessage();

            // 注册服务
            ApplicationDomainEx.ServiceManager.AddService(ViewServiceKeys.PAGE_TEMPLATE_SERVICE, this);
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {
            this.AddToPageGroupCommand = new VCommand(this.AddToPageGroup);
            this.OpenScenePageCommand = new VCommand(this.OpenScenePage);
            this.UpdateCommand = new VCommand(this.Update);
            this.DeleteCommand = new VCommand(this.Delete);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void InitMessage()
        {
            ApplicationDomainEx.MessageManager.Register<ProjectOpenMessage>(this, this.OnProjectOpenMessage);
            ApplicationDomainEx.MessageManager.Register<ProjectCloseMessage>(this, this.OnProjectCloseMessage);
            ApplicationDomainEx.MessageManager.Register<ProjectSaveMessage>(this, this.OnProjectSaveMessage);
        }

        // ======================================================================================
        // Field
        // ======================================================================================

        /// <summary>
        /// GH服务
        /// </summary>
        private GHService ghService = new GHService();

        /// <summary>
        /// 页服务
        /// </summary>
        private PageService pageService = new PageService();

        /// <summary>
        /// Viz命令服务
        /// </summary>
        private VizCommandService vizCommandService = new VizCommandService();

        /// <summary>
        /// Viz控制对象服务
        /// </summary>
        private VizCommandControlObjectService vizCommandControlObjectService = new VizCommandControlObjectService();

        /// <summary>
        /// 控制对象服务
        /// </summary>
        private ControlObjectService controlObjectService = new ControlObjectService();

        // ======================================================================================
        // Property
        // ======================================================================================

        #region IsEnabled -- 是否可用

        private bool isEnabled;
        /// <summary>
        /// 是否可用
        /// </summary>
        public bool IsEnabled
        {
            get { return isEnabled; }
            set { isEnabled = value; this.RaisePropertyChanged(nameof(IsEnabled)); }
        }

        #endregion

        #region IsLoading -- 是否正在加载

        private bool isLoading;
        /// <summary>
        /// 是否正在加载
        /// </summary>
        public bool IsLoading
        {
            get { return isLoading; }
            set { isLoading = value; this.RaisePropertyChanged(nameof(IsLoading)); }
        }

        #endregion

        #region SceneTemplateModels -- 场景模板模型集合

        private ObservableCollection<PageTemplateModel> sceneTemplateModels;
        /// <summary>
        /// 场景模板集合
        /// </summary>
        public ObservableCollection<PageTemplateModel> SceneTemplateModels
        {
            get { return sceneTemplateModels; }
            set { sceneTemplateModels = value; this.RaisePropertyChanged(nameof(SceneTemplateModels)); }
        }

        #endregion

        #region SelectedSceneTemplateModel -- 当前选中的场景模板模型

        private PageTemplateModel selectedSceneTemplateModel;
        /// <summary>
        /// 当前选中的场景模板模型
        /// </summary>
        public PageTemplateModel SelectedSceneTemplateModel
        {
            get { return selectedSceneTemplateModel; }
            set { selectedSceneTemplateModel = value; this.RaisePropertyChanged(nameof(SelectedSceneTemplateModel)); }
        }

        #endregion

        #region OtherTemplateModels -- 其他模板模型集合

        private ObservableCollection<PageTemplateModel> otherTemplateModels;
        /// <summary>
        /// 其他模板模型集合
        /// </summary>
        public ObservableCollection<PageTemplateModel> OtherTemplateModels
        {
            get { return otherTemplateModels; }
            set { otherTemplateModels = value; this.RaisePropertyChanged(nameof(OtherTemplateModels)); }
        }

        #endregion

        #region SelectedOtherTemplateModel -- 当前选中的其他模板模型

        private PageTemplateModel selectedOtherTemplateModel;
        /// <summary>
        /// 当前选中的其他模板模型
        /// </summary>
        public PageTemplateModel SelectedOtherTemplateModel
        {
            get { return selectedOtherTemplateModel; }
            set { selectedOtherTemplateModel = value; this.RaisePropertyChanged(nameof(SelectedOtherTemplateModel)); }
        }

        #endregion

        // ======================================================================================
        // Command
        // ======================================================================================

        #region AddToPageGroupCommand -- 场景模板添加至页分组命令

        /// <summary>
        /// 场景模板添加至页分组命令
        /// </summary>
        public VCommand AddToPageGroupCommand { get; set; }

        /// <summary>
        /// 场景模板添加至页分组
        /// </summary>
        private void AddToPageGroup()
        {
            if (this.SelectedSceneTemplateModel == null)
                return;

            IPageGroupService service = ApplicationDomainEx.ServiceManager.GetService<IPageGroupService>(ViewServiceKeys.PAGE_GROUP_SERVICE);
            if (service == null)
                return;

            service.AddPage(this.SelectedSceneTemplateModel);
        }

        #endregion

        #region OpenScenePageCommand -- 打开场景页命令

        /// <summary>
        /// 打开场景页命令
        /// </summary>
        public VCommand OpenScenePageCommand { get; set; }

        /// <summary>
        /// 打开场景页
        /// </summary>
        private void OpenScenePage()
        {
            if (this.SelectedSceneTemplateModel == null)
                return;

            if (ApplicationDomainEx.CurrentPage == this.SelectedSceneTemplateModel)
                return;

            ApplicationDomainEx.CurrentPage = this.SelectedSceneTemplateModel;

            PageOpenMessage msg = new PageOpenMessage();
            msg.Page = this.SelectedSceneTemplateModel;
            ApplicationDomainEx.MessageManager.Send(msg);
        }

        #endregion

        #region UpdateCommand -- 更新命令

        /// <summary>
        /// 更新命令
        /// </summary>
        public VCommand UpdateCommand { get; set; }

        /// <summary>
        /// 更新
        /// </summary>
        private void Update()
        {
            if (this.SelectedSceneTemplateModel == null)
                return;

            IFieldTreeService service = ApplicationDomainEx.ServiceManager.GetService<IFieldTreeService>(ViewServiceKeys.FIELD_TREE_SERVICE);
            if (service == null)
                return;

            this.IsLoading = true;

            ThreadHelper.SafeRun(action: () =>
            {
                // 重新加载版子
                this.vizCommandService.Reload(ApplicationDomainEx.PreviewConn, this.SelectedSceneTemplateModel.ScenePath);

                // 获取控制对象
                ControlObjectModel controlObject = this.vizCommandControlObjectService.GetControlObject(ApplicationDomainEx.PreviewConn);

                // 将控制字段写入数据库
                this.controlObjectService.SaveControlFields(this.SelectedSceneTemplateModel, controlObject.AllFiledNodes); ;

                // 如果当前打开的项目是该模板，那么从新打开
                if (ApplicationDomainEx.CurrentPage == this.SelectedSceneTemplateModel)
                {
                    WPFHelper.Invoke(() =>
                    {
                        this.OpenScenePage();
                    });
                }
            },
            final: () =>
            {
                WPFHelper.BeginInvoke(() =>
                {
                    this.IsLoading = false;
                });
            });
        }

        #endregion

        #region DeleteCommand -- 删除命令

        /// <summary>
        /// 删除命令
        /// </summary>
        public VCommand DeleteCommand { get; set; }

        /// <summary>
        /// 删除
        /// </summary>
        private void Delete()
        {
            if (this.SelectedSceneTemplateModel == null)
                return;

            if (DXMessageBox.Show($"是否删除模板【{this.SelectedSceneTemplateModel.Scene}】?", "提示", System.Windows.MessageBoxButton.YesNo) != System.Windows.MessageBoxResult.Yes)
                return;

            this.SceneTemplateModels.Remove(this.SelectedSceneTemplateModel);
        }

        #endregion

        // ======================================================================================
        // Message
        // ======================================================================================

        /// <summary>
        /// 项目打开消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProjectOpenMessage(ProjectOpenMessage msg)
        {
            this.IsEnabled = true;
            this.SceneTemplateModels = null;
            this.SelectedSceneTemplateModel = null;
            this.SelectedOtherTemplateModel = null;
            this.IsLoading = true;

            ThreadHelper.SafeRun(() =>
            {
                // 场景模板
                var sceneTemplateModels = this.pageService.LoadPageTemplates().ToObservableCollection();

                // 其他模板
                var otherTemplateModels = new ObservableCollection<PageTemplateModel>();

                // 命令模板
                PageTemplateModel cmd = new PageTemplateModel();
                cmd.PageType = PageType.Command;
                cmd.Remark = "命令模板";
                cmd.EngineType = EngineType.Viz;
                cmd.ThumbnailBitmap = VIZ.Package.Module.Resource.Properties.Resources.cmd_template;
                otherTemplateModels.Add(cmd);

                // 更新界面
                WPFHelper.BeginInvoke(() =>
                {
                    this.SceneTemplateModels = sceneTemplateModels;
                    this.OtherTemplateModels = otherTemplateModels;

                    this.IsLoading = false;
                });

                // 获取场景模板缩略图
                foreach (var model in sceneTemplateModels)
                {
                    var bitmap = ghService.GetImage(model.ThumbnailUri);

                    WPFHelper.BeginInvoke(() =>
                    {
                        model.ThumbnailBitmap = bitmap;
                    });
                }
            });
        }

        /// <summary>
        /// 项目关闭消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProjectCloseMessage(ProjectCloseMessage msg)
        {
            this.IsEnabled = false;
            this.IsLoading = false;
            this.SceneTemplateModels = null;
            this.SelectedSceneTemplateModel = null;
            this.SelectedOtherTemplateModel = null;
        }

        /// <summary>
        /// 项目保存消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProjectSaveMessage(ProjectSaveMessage msg)
        {
            this.pageService.SavePageTemplates(this.SceneTemplateModels);
        }

        // ======================================================================================
        // Public Function
        // ======================================================================================

        /// <summary>
        /// 添加场景模板
        /// </summary>
        /// <param name="fileModel">文件模型</param>
        public void AddSceneTemplate(GHResourceFileModel fileModel)
        {
            // 如果场景中已经拥有该模板，那么不处理
            if (this.SceneTemplateModels.Any(p => p.Scene == fileModel.Name))
                return;

            PageTemplateModel model = new PageTemplateModel();
            model.TemplateID = Guid.NewGuid();
            model.Scene = fileModel.Name;
            model.ScenePath = fileModel.Path;
            model.Layer = VizLayer.MAIN_LAYER;
            model.PageType = PageType.Scene;
            model.EngineType = EngineType.Viz;
            model.ThumbnailUri = fileModel.Thumbnail;
            model.ThumbnailBitmap = fileModel.ThumbnailBitmap;
            // 获取缩略图
            if (model.ThumbnailBitmap == null)
            {
                ThreadHelper.SafeRun(() =>
                {
                    var bitmap = this.ghService.GetImage(model.ThumbnailUri);

                    WPFHelper.BeginInvoke(() =>
                    {
                        model.ThumbnailBitmap = bitmap;
                    });
                });
            }

            this.SceneTemplateModels.Add(model);
        }
    }
}