﻿using DevExpress.Data;
using DevExpress.Mvvm.POCO;
using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.Package.Domain;
using VIZ.Package.Plugin;
using VIZ.Package.Service;

namespace VIZ.Package.Module
{
    /// <summary>
    /// 包装任务视图模型
    /// </summary>
    public class PackageTaskViewModel : ViewModelBase, IPackageTaskService, IPackageTaskInterface
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(PackageTaskViewModel));

        public PackageTaskViewModel()
        {
            // 初始化命令
            this.InitCommand();

            // 注册服务
            ApplicationDomainEx.ServiceManager.AddService(ViewServiceKeys.PACKAGE_TASK_SERVICE, this);
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {

        }

        // ==============================================================
        // Service & Controller
        // ==============================================================

        /// <summary>
        /// Viz引擎命令服务
        /// </summary>
        private VizCommandService vizCommandService = new VizCommandService();

        /// <summary>
        /// Viz引擎控制对象服务
        /// </summary>
        private VizCommandControlObjectService vizCommandControlObjectService = new VizCommandControlObjectService();

        // ==============================================================
        // Property
        // ==============================================================

        #region ItemsSource -- 任务源

        private ObservableCollection<PackageTaskModel> itemsSource = new ObservableCollection<PackageTaskModel>();
        /// <summary>
        /// 任务源
        /// </summary>
        public ObservableCollection<PackageTaskModel> ItemsSource
        {
            get { return itemsSource; }
            set { itemsSource = value; this.RaisePropertyChanged(nameof(ItemsSource)); }
        }

        #endregion

        // ==============================================================
        // Command
        // ==============================================================


        // ==============================================================
        // Public Function
        // ==============================================================

        /// <summary>
        /// 注册任务
        /// </summary>
        /// <param name="task">任务</param>
        public void Register(PackageTaskModel task)
        {
            lock (this.ItemsSource)
            {
                if (this.ItemsSource.Contains(task))
                    return;

                task.PackageTaskInterface = this;

                this.ItemsSource.Add(task);
            }
        }

        /// <summary>
        /// 取消任务
        /// </summary>
        /// <param name="task">任务</param>
        public void Cancel(PackageTaskModel task)
        {
            lock (this.ItemsSource)
            {
                if (this.ItemsSource.Contains(task))
                {
                    task.IsEnabled = false;

                    this.ItemsSource.Remove(task);
                }
            }
        }

        /// <summary>
        /// 预览更新
        /// </summary>
        /// <param name="task">任务</param>
        public void PreviewUpdate(PackageTaskModel task)
        {
            try
            {
                task.PreviewUpdateAction(ApplicationDomainEx.PreviewConn);
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }
        }

        /// <summary>
        /// 上版更新
        /// </summary>
        /// <param name="task">任务</param>
        public void TakeUpdate(PackageTaskModel task)
        {
            IPluginService pluginService = ApplicationDomainEx.ServiceManager.GetService<IPluginService>(ViewServiceKeys.PLUGIN_SERVICE);
            if (pluginService == null)
                return;

            PageModel page = pluginService.GetPageModelFromView(task.View);
            if (page == null)
                return;

            ConnGroupModel group = ApplicationDomainEx.ConnGroups.FirstOrDefault(p => p.GroupID == page.ConnGroupID);
            if (group == null)
                return;

            foreach (var item in group.Items)
            {
                if (!item.IsEnabled || !item.IsConnected)
                    continue;

                try
                {
                    task.TakeUpdateAction(item);
                }
                catch (Exception ex)
                {
                    log.Error(ex);
                }
            }
        }

        /// <summary>
        /// 销毁
        /// </summary>
        /// <param name="task">任务</param>
        public void Dispose(PackageTaskModel task)
        {
            this.Cancel(task);
        }
    }
}