﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Documents;
using VIZ.Framework.Core;
using VIZ.TVP.Golf.Domain;

namespace VIZ.TVP.Golf.Module
{
    /// <summary>
    /// 分组洞信息视图模型
    /// </summary>
    public class GroupHoleInfoViewModel : PackageViewModelBase
    {
        public GroupHoleInfoViewModel()
        {
            // 初始化命令
            this.InitCommand();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {
            this.SummaryCommand = new VCommand(this.Summary);
        }

        // ===================================================================================
        // Field
        // ===================================================================================

        // ===================================================================================
        // Property
        // ===================================================================================

        #region SelectedRound -- 选中的轮次

        private int selectedRound;
        /// <summary>
        /// 选中的轮次
        /// </summary>
        public int SelectedRound
        {
            get { return selectedRound; }
            set { selectedRound = value; this.RaisePropertyChanged(nameof(SelectedRound)); }
        }

        #endregion

        #region GroupPickerPanelModel -- 分组选择面板模型

        private GroupPickerPanelModel groupPickerPanelModel = new GroupPickerPanelModel();
        /// <summary>
        /// 分组选择面板模型
        /// </summary>
        public GroupPickerPanelModel GroupPickerPanelModel
        {
            get { return groupPickerPanelModel; }
            set { groupPickerPanelModel = value; this.RaisePropertyChanged(nameof(GroupPickerPanelModel)); }
        }

        #endregion

        #region GroupHoleTempModels -- 分组临时模型集合

        private ObservableCollection<GroupHoleTempModel> groupHoleTempModels;
        /// <summary>
        /// 分组临时模型
        /// </summary>
        public ObservableCollection<GroupHoleTempModel> GroupHoleTempModels
        {
            get { return groupHoleTempModels; }
            set { groupHoleTempModels = value; this.RaisePropertyChanged(nameof(GroupHoleTempModels)); }
        }

        #endregion

        #region First18HoleStrokes -- 前18洞总杆数

        private string first18HoleStrokes;
        /// <summary>
        /// 前18洞总杆数
        /// </summary>
        public string First18HoleStrokes
        {
            get { return first18HoleStrokes; }
            set { first18HoleStrokes = value; this.RaisePropertyChanged(nameof(First18HoleStrokes)); }
        }

        #endregion

        #region Secend18HoleStrokes -- 后18洞中杆数

        private string secend18HoleStrokes;
        /// <summary>
        /// 后18洞中杆数
        /// </summary>
        public string Secend18HoleStrokes
        {
            get { return secend18HoleStrokes; }
            set { secend18HoleStrokes = value; this.RaisePropertyChanged(nameof(Secend18HoleStrokes)); }
        }

        #endregion

        #region TotalHoleStrokes -- 总计杆数

        private string totalHoleStrokes;
        /// <summary>
        /// 总计杆数
        /// </summary>
        public string TotalHoleStrokes
        {
            get { return totalHoleStrokes; }
            set { totalHoleStrokes = value; this.RaisePropertyChanged(nameof(TotalHoleStrokes)); }
        }

        #endregion

        // ===================================================================================
        // Command
        // ===================================================================================

        #region SendCommand -- 发送命令

        /// <summary>
        /// 执行发送命令
        /// </summary>
        protected override void Send()
        {

        }

        #endregion

        #region LoadLocalDataCommand -- 加载本地数据命令

        /// <summary>
        /// 加载本地数据
        /// </summary>
        protected override void LoadLocalData()
        {
            RealDataWindow window = new RealDataWindow();
            window.ShowDialog();

            RealDataViewModel vm = window.realDataView.DataContext as RealDataViewModel;
            if (vm == null)
                return;

            if (!vm.IsEnter)
                return;

            List<PlayerRealModel> list = this.realDataService.LoadPlayerRealModelFormLocal(vm.SelectedFile.FileName);

            this.UpdateGroupHoleTempModels(list);
        }

        #endregion

        #region LoadRemoteDataCommand -- 加载远程数据命令

        /// <summary>
        /// 加载远程数据
        /// </summary>
        protected override void LoadRemoteData()
        {
            Task.Run(() =>
            {
                string fileName = this.realDataService.DownLoadData(INTERFACE_TOURNAMENT);

                if (string.IsNullOrWhiteSpace(fileName))
                {
                    MessageBox.Show("加载远程数据失败！");

                    return;
                }

                WPFHelper.BeginInvoke(() =>
                {
                    List<PlayerRealModel> list = this.realDataService.LoadPlayerRealModelFormLocal(fileName);

                    this.UpdateGroupHoleTempModels(list);
                });
            });
        }

        #endregion

        #region SummaryCommand -- 汇总计算

        /// <summary>
        /// 汇总计算
        /// </summary>
        public VCommand SummaryCommand { get; set; }

        /// <summary>
        /// 汇总
        /// </summary>
        private void Summary()
        {
            if (this.GroupHoleTempModels == null || this.GroupHoleTempModels.Count == 0)
            {
                this.First18HoleStrokes = null;
                this.Secend18HoleStrokes = null;
                this.TotalHoleStrokes = null;

                return;
            }

            int first18 = this.GroupHoleTempModels.Take(18).Sum(p => ValueHelper.ConvertValue<int>(p.TotalStrokes));
            int secend18 = this.GroupHoleTempModels.Skip(18).Take(18).Sum(p => ValueHelper.ConvertValue<int>(p.TotalStrokes));

            this.First18HoleStrokes = first18.ToString();
            this.Secend18HoleStrokes = secend18.ToString();
            this.TotalHoleStrokes = (first18 + secend18).ToString();
        }

        #endregion

        // ===================================================================================
        // Action
        // ===================================================================================

        /// <summary>
        /// 更新数据
        /// </summary>
        /// <param name="list">真实球员模型</param>
        private void UpdateGroupHoleTempModels(List<PlayerRealModel> list)
        {
            var players = ApplicationDomainEx.PlayerInfos.Where(p => p.Group == this.GroupPickerPanelModel.SelectedGroupInfo.Group);

            ObservableCollection<GroupHoleTempModel> groupHoleTempModels = new ObservableCollection<GroupHoleTempModel>();

            foreach (HoleInfoModel holeInfo in ApplicationDomainEx.HoleInfos)
            {
                GroupHoleTempModel model = new GroupHoleTempModel();
                model.HoleID = holeInfo.HoleID.ToString();
                model.Par = holeInfo.Par;

                int total = 0;

                foreach (PlayerInfoModel player in players)
                {
                    PlayerRealModel player_real = list.FirstOrDefault(p => p.PlayerID == player.PlayerID);
                    if (player_real == null)
                        continue;

                    RoundRealModel round_real = player_real.Rounds.FirstOrDefault(p => p.No == this.SelectedRound);
                    if (round_real == null)
                        continue;

                    total += round_real.Today;
                    model.TotalStrokesDetail += $"{player.Name}: {round_real.Today} | ";
                }

                model.TotalStrokes = total.ToString();

                groupHoleTempModels.Add(model);
            }

            this.GroupHoleTempModels = groupHoleTempModels;

            // 统计
            this.Summary();
        }
    }
}
