﻿using log4net;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Media;
using VIZ.Framework.Core;
using VIZ.TVP.Golf.Domain;
using VIZ.TVP.Golf.Service;
using VIZ.TVP.Golf.Storage;

namespace VIZ.TVP.Golf.Module
{
    /// <summary>
    /// 包装视图模型基类
    /// </summary>
    public abstract class PackageViewModelBase : ViewModelBase
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(PackageViewModelBase));

        public PackageViewModelBase()
        {
            this.SendCommand = new VCommand(this.Send);
            this.LoadLocalDataCommand = new VCommand(this.LoadLocalData);
            this.LoadRemoteDataCommand = new VCommand(this.LoadRemoteDataShell);
        }

        /// <summary>
        /// 赛事信息接口
        /// </summary>
        protected readonly static string INTERFACE_TOURNAMENT = ApplicationDomainEx.IniStorage.GetValue<InterfaceConfig, string>(p => p.INTERFACE_TOURNAMENT);

        // ===================================================================================
        // Field
        // ===================================================================================

        /// <summary>
        /// 实时数据服务
        /// </summary>
        protected IRealDataService realDataService = new RealDataService();

        // ===================================================================================
        // Property
        // ===================================================================================

        #region IsLoadRemoteDataEnabled -- 加载远程数据是否可用

        private bool isLoadRemoteDataEnabled = true;
        /// <summary>
        /// 加载远程数据是否可用
        /// </summary>
        public bool IsLoadRemoteDataEnabled
        {
            get { return isLoadRemoteDataEnabled; }
            set { isLoadRemoteDataEnabled = value; this.RaisePropertySaveChanged(nameof(IsLoadRemoteDataEnabled)); }
        }

        #endregion

        // ===================================================================================
        // Command
        // ===================================================================================

        #region SendCommand -- 发送命令

        /// <summary>
        /// 发送命令
        /// </summary>
        public VCommand SendCommand { get; set; }

        /// <summary>
        /// 发送指令
        /// </summary>
        protected abstract void Send();

        #endregion

        #region LoadLocalDataCommand -- 加载本地数据命令

        /// <summary>
        /// 加载本地数据命令
        /// </summary>
        public VCommand LoadLocalDataCommand { get; set; }

        /// <summary>
        /// 加载本地数据
        /// </summary>
        protected abstract void LoadLocalData();

        #endregion

        #region LoadRemoteDataCommand -- 加载远程数据命令

        /// <summary>
        /// 加载远程数据命令
        /// </summary>
        public VCommand LoadRemoteDataCommand { get; set; }

        /// <summary>
        /// 执行加载远程数据壳
        /// </summary>
        private void LoadRemoteDataShell()
        {
            try
            {
                this.IsLoadRemoteDataEnabled = false;

                ApplicationDomainEx.DelayManager.Wait("PackageViewModelBase.LoadRemoteDataShell", 2, () =>
                {
                    this.IsLoadRemoteDataEnabled = true;
                });

                this.LoadRemoteData();
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }
        }

        /// <summary>
        /// 加载远程数据
        /// </summary>
        protected abstract void LoadRemoteData();

        #endregion

        // ===================================================================================
        // Action
        // ===================================================================================

        /// <summary>
        /// 更新球员临时数据
        /// </summary>
        /// <param name="tempModel">临时模型</param>
        /// <param name="realModels">实时模型集合</param>
        protected void UpdatePlayerTempModel(PlayerTempModel tempModel, List<PlayerRealModel> realModels)
        {
            if (tempModel == null || realModels == null || tempModel.PlayerID <= 0 || realModels.Count == 0)
                return;

            PlayerRealModel realModel = realModels.FirstOrDefault(p => p.PlayerID == tempModel.PlayerID);

            if (realModel != null)
            {
                tempModel.Strokes = realModel.Strokes;
            }
        }
    }
}
