﻿using DevExpress.ClipboardSource.SpreadsheetML;
using DevExpress.Xpf.Core;
using DevExpress.Xpf.Editors;
using DevExpress.Xpf.Grid;
using DevExpress.Xpf.Utils.About;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Dynamic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Markup;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using VIZ.TVP.Domain;

namespace VIZ.TVP.Common
{
    /// <summary>
    /// 三元编辑器
    /// </summary>
    [TemplatePart(Name = "PART_X", Type = typeof(LeftRightTextEdit))]
    [TemplatePart(Name = "PART_Y", Type = typeof(LeftRightTextEdit))]
    [TemplatePart(Name = "PART_Z", Type = typeof(LeftRightTextEdit))]
    public class TripletEdit : Control
    {
        static TripletEdit()
        {
            DefaultStyleKeyProperty.OverrideMetadata(typeof(TripletEdit), new FrameworkPropertyMetadata(typeof(TripletEdit)));
        }

        /// <summary>
        /// 鼠标移动值改变步长
        /// </summary>
        public const double MOUSE_MOVE_STEP = 0.1d;

        /// <summary>
        /// 鼠标最小移动
        /// </summary>
        public const double MOUSE_MIN_MOVE = 5;

        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            this.Loaded -= TripletEdit_Loaded;
            this.Loaded += TripletEdit_Loaded;

            this.DataContextChanged -= TripletEdit_DataContextChanged;
            this.DataContextChanged += TripletEdit_DataContextChanged;

            this.PART_X = this.Template.FindName("PART_X", this) as LeftRightTextEdit;
            this.PART_Y = this.Template.FindName("PART_Y", this) as LeftRightTextEdit;
            this.PART_Z = this.Template.FindName("PART_Z", this) as LeftRightTextEdit;

            if (this.PART_X != null)
            {
                this.PART_X.EditValueChanged -= PART_X_EditValueChanged;
                this.PART_X.EditValueChanged += PART_X_EditValueChanged;
            }

            if (this.PART_Y != null)
            {
                this.PART_Y.EditValueChanged -= PART_Y_EditValueChanged;
                this.PART_Y.EditValueChanged += PART_Y_EditValueChanged;
            }

            if (this.PART_Z != null)
            {
                this.PART_Z.EditValueChanged -= PART_Z_EditValueChanged;
                this.PART_Z.EditValueChanged += PART_Z_EditValueChanged;
            }
        }

        #region PART

        /// <summary>
        /// X 值输入框
        /// </summary>
        private LeftRightTextEdit PART_X;

        /// <summary>
        /// Y 值输入框
        /// </summary>
        private LeftRightTextEdit PART_Y;

        /// <summary>
        /// Z 值输入框
        /// </summary>
        private LeftRightTextEdit PART_Z;

        #endregion

        /// <summary>
        /// 是否处于初始化状态
        /// </summary>
        private bool isInInit;

        // ------------------------------------------------------------
        // 绑定值改变

        /// <summary>
        /// 加载完成
        /// </summary>
        private void TripletEdit_Loaded(object sender, RoutedEventArgs e)
        {
            this.InitTextEditValue();
        }

        /// <summary>
        /// 绑定值改变
        /// </summary>
        private void TripletEdit_DataContextChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            this.InitTextEditValue();
        }

        /// <summary>
        /// 初始化文本编辑器值
        /// </summary>
        private void InitTextEditValue()
        {
            EditGridCellData cellData = this.DataContext as EditGridCellData;
            if (cellData == null)
                return;

            this.isInInit = true;

            if (cellData.Row is VizControlObjectFieldModel)
            {
                VizControlObjectFieldModel field = cellData.Row as VizControlObjectFieldModel;
                double[] triplet = this.GetTripletValue(field.Value);

                this.PART_X.EditValue = triplet[0];
                this.PART_Y.EditValue = triplet[1];
                this.PART_Z.EditValue = triplet[2];
            }

            if (cellData.Row is ExpandoObject)
            {
                IDictionary<string, object> dic = cellData.Row as IDictionary<string, object>;

                object obj = dic[cellData.Column.FieldName];
                double[] triplet = this.GetTripletValue(obj == null ? null : obj.ToString());

                this.PART_X.EditValue = triplet[0];
                this.PART_Y.EditValue = triplet[1];
                this.PART_Z.EditValue = triplet[2];
            }

            this.isInInit = false;
        }

        // ------------------------------------------------------------
        // 值改变

        /// <summary>
        /// X 值改变时触发
        /// </summary>
        private void PART_X_EditValueChanged(object sender, EditValueChangedEventArgs e)
        {
            this.TextEditValueChanged(0, e.NewValue?.ToString());
        }

        /// <summary>
        /// Y 值改变时触发
        /// </summary>
        private void PART_Y_EditValueChanged(object sender, EditValueChangedEventArgs e)
        {
            this.TextEditValueChanged(1, e.NewValue?.ToString());
        }

        /// <summary>
        /// Z 值改变时触发
        /// </summary>
        private void PART_Z_EditValueChanged(object sender, EditValueChangedEventArgs e)
        {
            this.TextEditValueChanged(2, e.NewValue?.ToString());
        }

        /// <summary>
        /// 输入文本值改变时触发
        /// </summary>
        private void TextEditValueChanged(int index, string value)
        {
            if (this.isInInit)
                return;

            EditGridCellData cellData = this.DataContext as EditGridCellData;
            if (cellData == null)
                return;

            double.TryParse(value, out double v);

            if (cellData.Row is VizControlObjectFieldModel)
            {
                VizControlObjectFieldModel field = cellData.Row as VizControlObjectFieldModel;
                double[] triplet = this.GetTripletValue(field.Value);
                triplet[index] = v;

                field.Value = string.Join(" ", triplet);
            }

            if (cellData.Row is ExpandoObject)
            {
                IDictionary<string, object> dic = cellData.Row as IDictionary<string, object>;

                object obj = dic[cellData.Column.FieldName];
                double[] triplet = this.GetTripletValue(obj?.ToString());
                triplet[index] = v;

                dic[cellData.Column.FieldName] = string.Join(" ", triplet);
            }
        }

        // ------------------------------------------------------------
        // Other

        /// <summary>
        /// 获取三元值
        /// </summary>
        /// <param name="value">值</param>
        /// <returns>三元值列表</returns>
        private double[] GetTripletValue(string value)
        {
            double[] result = new double[] { 0, 0, 0 };

            if (string.IsNullOrWhiteSpace(value))
                return result;

            string[] pars = value.Split(' ');
            for (int i = 0; i < pars.Length; i++)
            {
                if (!double.TryParse(pars[i], out double d))
                    continue;

                result[i] = d;
            }

            return result;
        }
    }
}
