﻿using DevExpress.Mvvm.POCO;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Dynamic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using VIZ.Framework.Core;
using VIZ.Framework.Plugin;
using VIZ.TVP.Domain;
using VIZ.TVP.Service;
using VIZ.TVP.Storage;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// Control List视图模型
    /// </summary>
    public class ControlListViewModel : PluginViewModelBase, IControlObjectSupport
    {
        public ControlListViewModel()
        {
            // 初始化控制器
            this.initController();

            // 初始化命令
            this.initCommand();

            // 初始化消息
            this.initMessage();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void initCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
            this.RefreshControlObjectFieldListCommand = new VCommand(this.RefreshControlObjectFieldList);
            this.RefreshControlObjectFromVizCommand = new VCommand(this.RefreshControlObjectFromViz);
        }

        /// <summary>
        /// 初始化控制器
        /// </summary>
        private void initController()
        {
            this.controlObjectController = new ControlObjectController(this);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void initMessage()
        {
            ApplicationDomainEx.MessageManager.Register<ProjectChangedMessage>(this, this.OnProjectChangedMessage);
            ApplicationDomainEx.MessageManager.Register<ProgramListItemChangedMessage>(this, this.OnProgramListItemChangedMessage);
            ApplicationDomainEx.MessageManager.Register<ProgramListItemInitedMessage>(this, this.OnProgramListItemInitedMessage);
            ApplicationDomainEx.MessageManager.Register<VizControlObjectFieldValueChangedMessage>(this, this.OnVizControlObjectFieldValueChangedMessage);
        }

        // ==================================================================================
        // Service & Control
        // ==================================================================================

        /// <summary>
        /// ControlObject命令服务
        /// </summary>
        private IVizCommandControlObjectService vizCommandControlObjectService = new VizCommandControlObjectService();

        /// <summary>
        /// 控制对象控制器
        /// </summary>
        private ControlObjectController controlObjectController;

        // ==================================================================================
        // Property
        // ==================================================================================

        #region IsLoading -- 是否正在加载

        private bool isLoading;
        /// <summary>
        /// 是否正在加载
        /// </summary>
        public bool IsLoading
        {
            get { return isLoading; }
            set { isLoading = value; this.RaisePropertyChanged(nameof(IsLoading)); }
        }

        #endregion

        #region ControlObjectList -- 控制对象列表

        private ObservableCollection<VizControlObjectModel> controlObjectList;
        /// <summary>
        /// 控制对象列表
        /// </summary>
        public ObservableCollection<VizControlObjectModel> ControlObjectList
        {
            get { return controlObjectList; }
            set { controlObjectList = value; this.RaisePropertyChanged(nameof(ControlObjectList)); }
        }

        #endregion

        #region SelectedControlObject -- 当前选中的控制对象

        private VizControlObjectModel selectedControlObject;
        /// <summary>
        /// 当前选中的控制对象
        /// </summary>
        public VizControlObjectModel SelectedControlObject
        {
            get { return selectedControlObject; }
            set { selectedControlObject = value; this.RaisePropertyChanged(nameof(SelectedControlObject)); }
        }

        #endregion

        #region SelectedControlObjectField -- 当前选中的控制对象字段

        private VizControlObjectFieldModel selectedControlObjectField;
        /// <summary>
        /// 当前选中的控制对象字段
        /// </summary>
        public VizControlObjectFieldModel SelectedControlObjectField
        {
            get { return selectedControlObjectField; }
            set
            {
                selectedControlObjectField = value;
                this.RaisePropertyChanged(nameof(SelectedControlObjectField));
                this.controlObjectController.UpdateSelectedControlObjectField(this.GetView<ControlListView>());
            }
        }

        #endregion

        #region ListValue -- List类型数据列表

        private List<ExpandoObject> listValue;
        /// <summary>
        /// List类型数据列表
        /// </summary>
        public List<ExpandoObject> ListValue
        {
            get { return listValue; }
            set { listValue = value; this.RaisePropertyChanged(nameof(ListValue)); }
        }

        #endregion

        #region ListValueColumns -- List类型数据列信息

        private List<GridColumnDefinition> listValueColumns;
        /// <summary>
        /// List类型数据列信息
        /// </summary>
        public List<GridColumnDefinition> ListValueColumns
        {
            get { return listValueColumns; }
            set { listValueColumns = value; this.RaisePropertyChanged(nameof(ListValueColumns)); }
        }

        #endregion

        // ==================================================================================
        // Command
        // ==================================================================================

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {

        }

        #endregion

        #region RefreshControlObjectFieldListCommand -- 刷新控制对象字段列表命令

        /// <summary>
        /// 刷新控制对象字段列表命令
        /// </summary>
        public VCommand RefreshControlObjectFieldListCommand { get; set; }

        /// <summary>
        /// 刷新控制对象字段列表
        /// </summary>
        private void RefreshControlObjectFieldList()
        {
            if (this.SelectedControlObjectField == null || this.SelectedControlObjectField.Type != VizControlObjectFieldType.list)
                return;

            this.SelectedControlObjectField.IsXmlValueLoaded = false;
            this.controlObjectController.UpdateSelectedControlObjectField(this.GetView<ControlListView>());
        }

        #endregion

        #region RefreshControlObjectFromVizCommand -- 从VIZ中获取控制对象命令

        /// <summary>
        /// 从VIZ中获取控制对象命令
        /// </summary>
        public VCommand RefreshControlObjectFromVizCommand { get; set; }

        /// <summary>
        /// 从VIZ中获取控制对象
        /// </summary>
        private void RefreshControlObjectFromViz()
        {
            if (ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            // 从Viz中获取控制对象
            this.controlObjectController.GetControlObjectFromViz(ApplicationDomainEx.SelectionManager.ProgramListItemModel, false);
        }

        #endregion

        // ==================================================================================
        // Message
        // ==================================================================================

        /// <summary>
        /// 项目改变消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProjectChangedMessage(ProjectChangedMessage msg)
        {
            this.HasProjectDomain = msg.NewProjectDomain != null;
            this.ControlObjectList = null;
            this.SelectedControlObject = null;
            this.SelectedControlObjectField = null;
            this.ListValue = null;
            this.ListValueColumns = null;
        }

        /// <summary>
        /// 节目单项目改变消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProgramListItemChangedMessage(ProgramListItemChangedMessage msg)
        {
            if (msg.ProgramListItemModel == null)
            {
                this.ControlObjectList = null;
                this.SelectedControlObject = null;
                return;
            }

            this.ControlObjectList = msg.ProgramListItemModel.ControlObjectList;
            this.SelectedControlObject = this.ControlObjectList.FirstOrDefault();
        }

        /// <summary>
        /// 处理节目单项初始化完成消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProgramListItemInitedMessage(ProgramListItemInitedMessage msg)
        {
            // 如果已经加载过控制对象，那么不再加载
            if (msg.ProgramListItemModel.IsLoadedControlObjectList)
                return;

            // 从Viz中获取控制对象
            this.controlObjectController.GetControlObjectFromViz(msg.ProgramListItemModel, true);
        }

        /// <summary>
        /// 控制对象值改变消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnVizControlObjectFieldValueChangedMessage(VizControlObjectFieldValueChangedMessage msg)
        {
            if (msg.FieldModel.ProgramListItemModel != ApplicationDomainEx.SelectionManager.ProgramListItemModel)
                return;

            TVPConnectionModel connection = ApplicationDomainEx.TVPConnectionManager.LocalConnection;
            string treeNodePath = msg.FieldModel.VizControlObjectModel.TreeNodeInfo.NodeFullPath;
            this.vizCommandControlObjectService.SetControlObjectValue(connection, Storage.VizSceneLayerEnum.MAIN_SCENE, treeNodePath, msg.FieldModel.FieldIdentifier, msg.FieldModel.Value);
        }

        // ==================================================================================
        // Public Function
        // ==================================================================================

        /// <summary>
        /// 销毁
        /// </summary>
        public override void Dispose()
        {

        }

    }
}
