﻿using DevExpress.Internal.WinApi.Windows.UI.Notifications;
using DevExpress.Mvvm.Xpf;
using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.IO.Pipes;
using System.Linq;
using System.Net.NetworkInformation;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Shapes;
using VIZ.Framework.Core;
using VIZ.Framework.Plugin;
using VIZ.TVP.Domain;
using VIZ.TVP.Service;
using VIZ.TVP.Storage;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// 本地资源视图模型
    /// </summary>
    public class LocalResourceViewModel : PluginViewModelBase, ILocalResourceFileSupport
    {


        /// <summary>
        /// 日志
        /// </summary>
        private static readonly ILog log = LogManager.GetLogger(typeof(VizResourceViewModel));



        public LocalResourceViewModel()
        {
            // 初始化控制器
            this.initController();

            // 初始化命令
            this.initCommand();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void initCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
            this.RefreshFolderCommand = new VCommand(this.RefreshFolder);
            this.RefreshFileCommand = new VCommand(this.RefreshFile);
            this.FolderExpandCommand = new VCommand<DevExpress.Xpf.Grid.TreeList.NodeDoubleClickEventArgs>(this.FolderExpand);
            this.FileFilterCommand = new VCommand<ResourceFileType>(this.FileFilter);
            this.FileRowFilterCommand = new DevExpress.Mvvm.DelegateCommand<RowFilterArgs>(this.FileRowFilter);

            this.FileContextMenuOpendCommand = new VCommand(this.FileContextMenuOpend);

            this.AddProgramTemplateCommand = new VCommand(this.AddProgramTemplate, this.CanAddProgramTemplate);

            this.CopyFilePathCommand = new VCommand(this.CopyFilePath, this.CanCopyFilePath);
        }

        /// <summary>
        /// 初始化控制器
        /// </summary>
        private void initController()
        {
            this.localResourceFileController = new LocalResourceFileController(this);
        }

        // ==================================================================================
        // Property
        // ==================================================================================

        #region FolderModels -- 文件夹目录集合

        private ObservableCollection<GHResourceFolderModel> folderModels;
        /// <summary>
        /// 文件夹目录集合
        /// </summary>
        public ObservableCollection<GHResourceFolderModel> FolderModels
        {
            get { return folderModels; }
            set
            {
                folderModels = value;
                this.RaisePropertyChanged(nameof(FolderModels));
            }
        }

        #endregion

        #region SelectedFolderModel -- 当前选中的文件夹

        private GHResourceFolderModel selectedFolderModel;
        /// <summary>
        /// 当前选中的文件夹
        /// </summary>
        public GHResourceFolderModel SelectedFolderModel
        {
            get { return selectedFolderModel; }
            set
            {
                selectedFolderModel = value;
                this.RaisePropertyChanged(nameof(SelectedFolderModel));

                if (value == null) return;


                GetFile(value);

                localResourceFileController.UpdateFileModels(value);

            }
        }

        #endregion

        #region FileModels -- 文件集合

        private ObservableCollection<GHResourceFileModel> fileModels;
        /// <summary>
        /// 文件集合
        /// </summary>
        public ObservableCollection<GHResourceFileModel> FileModels
        {
            get { return fileModels; }
            set
            {

                fileModels = value;
                this.RaisePropertyChanged(nameof(FileModels));
            }
        }

        #endregion

        #region SelectedFileModel -- 选中的文件模型

        private GHResourceFileModel selectedFileModel;
        /// <summary>
        /// 选中的文件模型
        /// </summary>
        public GHResourceFileModel SelectedFileModel
        {
            get { return selectedFileModel; }
            set { selectedFileModel = value; this.RaisePropertyChanged(nameof(SelectedFileModel)); }
        }

        #endregion

        #region IsFolderLoading -- 是否正在加载文件夹

        private bool isFolderLoading;
        /// <summary>
        /// 是否正在加载文件夹
        /// </summary>
        public bool IsFolderLoading
        {
            get { return isFolderLoading; }
            set { isFolderLoading = value; this.RaisePropertyChanged(nameof(IsFolderLoading)); }
        }

        #endregion

        #region IsFileLoading -- 是否正在加载文件

        private bool isFileLoading;
        /// <summary>
        /// 是否正在加载文件
        /// </summary>
        public bool IsFileLoading
        {
            get { return isFileLoading; }
            set { isFileLoading = value; this.RaisePropertyChanged(nameof(IsFileLoading)); }
        }

        #endregion

        #region FilterResourceFileType -- 资源文件类型

        private ResourceFileType filterResourceFileType;
        /// <summary>
        /// 资源文件类型
        /// </summary>
        public ResourceFileType FilterResourceFileType
        {
            get { return filterResourceFileType; }
            set { filterResourceFileType = value; this.RaisePropertySaveChanged(nameof(FilterResourceFileType)); }
        }

        #endregion

        // ==================================================================================
        // Service & Controller
        // ==================================================================================

        /// <summary>
        /// VIZ资源文件控制器
        /// </summary>
        private LocalResourceFileController localResourceFileController;

        // ==================================================================================
        // Command
        // ==================================================================================

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {
            if (this.IsAlreadyLoaded)
                return;

            this.IsFolderLoading = true;

            // 刷新文件夹
            this.RefreshFolder();

            this.IsAlreadyLoaded = true;
        }

        #endregion

        #region RefreshFolderCommand -- 刷新文件夹命令

        /// <summary>
        /// 刷新文件夹命令
        /// </summary>
        public VCommand RefreshFolderCommand { get; set; }

        /// <summary>
        /// 刷新文件夹
        /// </summary>
        private void RefreshFolder()
        {
            this.IsFolderLoading = true;

            Task.Run(() =>
            {
                try
                {
                    // 获取电脑上的每个逻辑驱动
                    this.FolderModels = new ObservableCollection<GHResourceFolderModel>();
                    foreach (var drive in Directory.GetLogicalDrives())
                    {
                        FolderModels.Add(new GHResourceFolderModel()
                        {
                            Name = drive,
                            Path = drive.ToString(),
                        });
                    }
                    this.IsFolderLoading = false;
                }
                catch (Exception ex)
                {
                    log.Error(ex);
                }
            });
        }

        #endregion

        /// <summary>
        /// 从完整路径中查找文件或文件夹名称
        /// </summary>
        /// <param name="path">完整路径</param>
        /// <returns></returns>
        public static string GetFileFolderName(string path)
        {
            // 如果没有路径，则返回空
            if (string.IsNullOrEmpty(path))
                return string.Empty;

            // 使所有斜杠成反斜杠
            var normalizedPath = path.Replace('/', '\\');

            // 找到最后一个反斜杠就是路径
            var lastIndex = normalizedPath.LastIndexOf('\\');

            // 如果找不到反斜线，则返回路径本身
            if (lastIndex <= 0)
                return path;

            // 最后一个反斜杠后返回名称
            return path.Substring(lastIndex + 1);
        }



        #region RefreshFileCommand -- 刷新文件命令

        /// <summary>
        /// 刷新文件命令
        /// </summary>
        public VCommand RefreshFileCommand { get; set; }

        /// <summary>
        /// 刷新文件
        /// </summary>
        private void RefreshFile()
        {
            GHResourceFolderModel folder = this.SelectedFolderModel;

            if (folder == null)
                return;

            this.localResourceFileController.DisposeFileModels(folder);

            GetFile(folder);

            this.localResourceFileController.UpdateFileModels(this.SelectedFolderModel);
        }

        #endregion

        #region FolderExpandCommand -- 文件夹展开命令

        /// <summary>
        /// 文件夹展开命令
        /// </summary>
        public VCommand<DevExpress.Xpf.Grid.TreeList.NodeDoubleClickEventArgs> FolderExpandCommand { get; set; }


        /// <summary>
        /// 文件夹展开
        /// </summary>
        private void FolderExpand(DevExpress.Xpf.Grid.TreeList.NodeDoubleClickEventArgs e)
        {

            // 获取完整路径
            var fullPath = selectedFolderModel.Path;
            #region Get Folders

            // 为目录创建一个空白列表
            var directories = new List<String>();


            // 尝试从文件夹中获取目录 // try异常处理,忽略出现的任何问题          
            try
            {
                var dirs = Directory.GetDirectories(fullPath);
                {
                    if (dirs.Length > 0)
                        directories.AddRange(dirs);
                }
            }
            catch { }


            //selectedFolderModel.FolderNode = new ObservableCollection<LFResourceFolderModel>();

            selectedFolderModel.Children.Clear();

            // 为目录做迭代,添加目录
            directories.ForEach(directoryPath =>
            {
                selectedFolderModel.Children.Add(new GHResourceFolderModel()
                {
                    Name = GetFileFolderName(directoryPath),
                    Path = directoryPath,
                    FolderType = ResourceFolderType.Folder
                });
            });

            #endregion







            if (this.SelectedFolderModel == null || e.ChangedButton != System.Windows.Input.MouseButton.Left)
                return;

            this.SelectedFolderModel.IsExpand = !this.SelectedFolderModel.IsExpand;

            //var tempfolderModel = FolderModels;
            //var tempSelecFolder = SelectedFolderModel;
            //FolderModels = new ObservableCollection<LFResourceFolderModel>();
            //FolderModels = tempfolderModel;
            //SelectedFolderModel = tempSelecFolder;
            //FileModels = tempSelecFolder.Files;





            //if (FileModels != null)
            //{
            //    foreach (var fileModel in tempSelecFolder.Files)
            //    {
            //        if (IsImage(fileModel.Path))
            //        {
            //            fileModel.ThumbnailBitmap = ReadImageFile(fileModel.Path);
            //        }

            //        fileModel.FileType = ResourceFileType.IMAGE;
            //    }
            //}



        }


        /// <summary>
        /// 获取选中的文件
        /// </summary>
        /// <param name="value"></param>
        private void GetFile(GHResourceFolderModel value)
        {

            #region Get Files
            // 为目录创建一个空白文件
            var Files = new List<String>();



            // 尝试从文件夹中获取文件 // try异常处理,忽略出现的任何问题          
            try
            {
                var fs = Directory.GetFiles(value.Path);
                {
                    if (fs.Length > 0)
                        Files.AddRange(fs);
                }
            }
            catch { }

            List<GHResourceFileModel> files = new List<GHResourceFileModel>();
            // 为目录做迭代,添加文件
            Files.ForEach(filePath =>
            {
                var ghFileModel = new GHResourceFileModel();
                ghFileModel.Name = GetFileFolderName(filePath);
                ghFileModel.Path = filePath;

                files.Add(ghFileModel);

            });

            selectedFolderModel.Files = files.ToObservableCollection();
            #endregion
        }

        #endregion

        #region FileFilterCommand -- 文件过滤命令

        /// <summary>
        /// 文件过滤命令
        /// </summary>
        public VCommand<ResourceFileType> FileFilterCommand { get; set; }

        /// <summary>
        /// 文件过滤
        /// </summary>
        /// <param name="type"></param>
        private void FileFilter(ResourceFileType type)
        {
            this.FilterResourceFileType = type;

            LocalResourceView view = this.GetView<LocalResourceView>();
            if (view == null)
                return;

            view.fileGrid.RefreshData();
        }

        #endregion

        #region CopyFilePathCommand -- 拷贝文件路径命令

        /// <summary>
        /// 拷贝文件路径命令
        /// </summary>
        public VCommand CopyFilePathCommand { get; set; }

        /// <summary>
        /// 是否可以执行拷贝文件路径
        /// </summary>
        /// <returns>是否可以执行拷贝文件路径</returns>
        private bool CanCopyFilePath()
        {
            return this.SelectedFileModel != null;
        }

        /// <summary>
        /// 拷贝文件路径
        /// </summary>
        private void CopyFilePath()
        {
            if (this.SelectedFileModel == null)
                return;

            Clipboard.SetText($"{this.SelectedFileModel.FileType}*{this.SelectedFileModel.Path}");
        }

        #endregion

        #region FileContextMenuOpendCommand -- 文件右键菜单打开命令

        /// <summary>
        /// 文件右键菜单打开命令
        /// </summary>
        public VCommand FileContextMenuOpendCommand { get; set; }

        /// <summary>
        /// 文件右键菜单打开
        /// </summary>
        private void FileContextMenuOpend()
        {
            this.AddProgramTemplateCommand.RaiseCanExecute();
        }

        #endregion

        #region AddProgramTemplateCommand -- 添加节目模板命令

        /// <summary>
        /// 添加节目模板命令
        /// </summary>
        public VCommand AddProgramTemplateCommand { get; set; }

        /// <summary>
        /// 是否可以执行添加节目模板
        /// </summary>
        /// <returns>是否可以执行添加节目模板</returns>
        private bool CanAddProgramTemplate()
        {
            return this.SelectedFileModel != null && this.SelectedFileModel.FileType == ResourceFileType.SCENE && ApplicationDomainEx.CurrentProjectDomain != null;
        }

        /// <summary>
        /// 添加节目模板
        /// </summary>
        private void AddProgramTemplate()
        {
            if (this.SelectedFileModel == null || this.SelectedFileModel.FileType != ResourceFileType.SCENE)
                return;

            IProgramListViewService service = ApplicationDomainEx.ServiceManager.GetService<IProgramListViewService>(ServiceKeys.PROGRAM_LIST_VIEW_SERVICE);
            if (service == null)
                return;

            // service.AddSceneTemplate(this.SelectedFileModel);
        }

        #endregion

        #region FileRowFilterCommand -- 文件行筛选命令

        /// <summary>
        /// 文件行筛选命令, 该命令必须使用Dev的命令基类
        /// </summary>
        public DevExpress.Mvvm.DelegateCommand<RowFilterArgs> FileRowFilterCommand { get; set; }

        /// <summary>
        /// 文件行筛选
        /// </summary>
        /// <param name="e">赛选参数</param>
        private void FileRowFilter(RowFilterArgs e)
        {
            GHResourceFileModel fileModel = e.Item as GHResourceFileModel;
            if (fileModel == null)
            {
                e.Visible = false;
                return;
            }

            if (this.FilterResourceFileType == ResourceFileType.None)
            {
                e.Visible = true;
                return;
            }

            e.Visible = fileModel.FileType == this.FilterResourceFileType;
        }

        #endregion

        // ==================================================================================
        // Public Function
        // ==================================================================================

        /// <summary>
        /// 销毁
        /// </summary>
        public override void Dispose()
        {

        }
    }

}
