﻿using System;
using System.Collections.Generic;
using System.Dynamic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Web.UI.WebControls;
using System.Xml.Linq;
using VIZ.TVP.Domain;
using VIZ.TVP.Storage;

namespace VIZ.TVP.Service
{
    /// <summary>
    /// VIZ命令ControlObject服务
    /// </summary>
    public class VizCommandControlObjectService : IVizCommandControlObjectService
    {
        /// <summary>
        /// 获取控制对象XML数据
        /// </summary>
        /// <param name="xml">xml内容</param> 
        /// <param name="schema">定义信息</param>
        /// <param name="columns">列信息</param>
        /// <param name="items">数据源</param>
        /// <returns></returns>
        public void GetControlObjectXmlData(string xml, ControlObject_Schema_Node schema, out List<GridColumnDefinition> columns, out List<ExpandoObject> items)
        {
            columns = new List<GridColumnDefinition>();
            items = new List<ExpandoObject>();

            bool isCreatedColumns = false;

            using (System.IO.MemoryStream ms = new System.IO.MemoryStream(System.Text.Encoding.UTF8.GetBytes(xml)))
            {
                XElement element = XElement.Load(ms);

                ControlObject_Entry_Node root = new ControlObject_Entry_Node();
                root.FromXmlElement(element);

                foreach (var row in root.Elements)
                {
                    IDictionary<string, object> obj = new ExpandoObject();
                    ControlObject_Entry_Node data = row.Entrys.FirstOrDefault(p => p.Name == "data");
                    if (data == null)
                        continue;

                    foreach (var cell in data.Entrys)
                    {
                        obj[cell.Name] = cell.Value;

                        if (!isCreatedColumns)
                        {
                            ControlObject_Field_node node = schema.Fields.FirstOrDefault(p => p.Name == cell.Name);

                            GridColumnDefinition column = new GridColumnDefinition();
                            column.FieldName = cell.Name;
                            if (cell.Name == node?.Description)
                            {
                                column.Header = cell.Name;
                            }
                            else
                            {
                                column.Header = $"({node?.Description}){cell.Name}";
                            }

                            columns.Add(column);
                        }
                    }

                    isCreatedColumns = true;

                    items.Add(obj as ExpandoObject);
                }
            }
        }

        /// <summary>
        /// 获取控制对象XML数据
        /// </summary>
        /// <param name="items">控制对象数据</param>
        /// <returns>XML数据</returns>
        public string GetControlObjectXml(List<ExpandoObject> items)
        {
            ControlObject_Entry_Node root = new ControlObject_Entry_Node();
            foreach (IDictionary<string, object> row in items)
            {
                ControlObject_Element_Node element = new ControlObject_Element_Node();
                ControlObject_Entry_Node data = new ControlObject_Entry_Node();
                data.Name = "data";
                foreach (var kv in row)
                {
                    ControlObject_Entry_Node cell = new ControlObject_Entry_Node();
                    cell.Name = kv.Key;
                    cell.Value = kv.Value == null ? string.Empty : kv.Value.ToString();
                    data.Entrys.Add(cell);
                }
                element.Entrys.Add(data);
                root.Elements.Add(element);
            }

            XElement root_element = root.ToXmlElement();

            return $"<?xml version=\"1.0\"?>{root_element.ToString(SaveOptions.DisableFormatting)}";
        }

        /// <summary>
        /// 获取场景树节点信息
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="layer">场景图层</param>
        /// <returns>场景树节点信息</returns>
        public List<VizTreeNodeInfo> GetTreeNodeList(TVPConnectionModel connection, VizSceneLayerEnum layer)
        {
            List<VizTreeNodeInfo> list = new List<VizTreeNodeInfo>();

            string cmd = $"{layer}*TREE GET";
            string result = connection.EndpointManager.Request(cmd);
            List<string> nodes = result.Split('{').Select(p => p.Replace("}", "")).ToList();

            foreach (string node in nodes)
            {
                if (string.IsNullOrWhiteSpace(node))
                    continue;

                // 1 29073 object 1
                string[] pars = node.Split(' ');

                VizTreeNodeInfo info = new VizTreeNodeInfo();
                info.NodeFullPath = pars[0];
                info.NodeNum = pars[1];
                info.NodeName = pars[2];
                // pars[3] 有的节点有，有的节点没有

                list.Add(info);
            }

            return list;
        }

        /// <summary>
        /// 获取控制对象列表
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="programListItemModel">节目单项模型</param>
        /// <param name="layer">场景图层</param>
        /// <returns>控制对象列表</returns>
        public List<VizControlObjectModel> GetControlObjectList(TVPConnectionModel connection, ProgramListItemModel programListItemModel, VizSceneLayerEnum layer)
        {
            List<VizControlObjectModel> list = new List<VizControlObjectModel>();

            // Step 1. 获取场景树信息
            List<VizTreeNodeInfo> treeNodeList = this.GetTreeNodeList(connection, layer);

            // Step 2. 获取控制对象ID集合
            List<string> ids = this.GetControlObjectIds(connection, layer);

            // Step 3. 获取控制对象信息, 只获取第一个控制对象
            foreach (string id in new string[] { ids.FirstOrDefault() })
            {
                if (string.IsNullOrWhiteSpace(id))
                    continue;

                VizControlObjectModel obj = new VizControlObjectModel();
                obj.ProgramListItemModel = programListItemModel;
                VizTreeNodeInfo info = treeNodeList.FirstOrDefault(p => id == $"#{p.NodeNum}");
                if (info == null)
                    continue;

                obj.TreeNodeInfo = info;
                obj.TreeNodeName = info.NodeName;
                obj.TreeNodePath = info.NodeFullPath;
                obj.Description = this.GetControlObjectParameter(connection, layer, info.NodeNum, VizControlObjectParameters.description);
                obj.UseAllDirectors = this.GetControlObjectParameter(connection, layer, info.NodeNum, VizControlObjectParameters.use_all_directors) == "1";

                string str = this.GetControlObjectResult(connection, layer, info.NodeNum);
                string[] lines = str.Split('\n');
                // BTNAME:#10245*GEOM*TEXT:text:::-1:BTNAME:single_line, location_id=#10245, location=2/3/5
                foreach (string line in lines)
                {
                    if (string.IsNullOrWhiteSpace(line))
                        continue;

                    string[] pars = line.Split(':');
                    VizControlObjectFieldModel field = new VizControlObjectFieldModel(new VizControlObjectFieldEntity());
                    field.ProgramListItemModel = programListItemModel;
                    field.VizControlObjectModel = obj;
                    field.FieldIdentifier = pars[0];
                    field.TreeNodePath = obj.TreeNodePath;
                    field.PropertyPath = pars[1];
                    field.Type = this.GetControlObjectFieldType(pars[2]);
                    if (field.Type == VizControlObjectFieldType.list)
                    {
                        field.FieldSchema = new ControlObject_Field_node();
                        using (System.IO.MemoryStream ms = new System.IO.MemoryStream(System.Text.Encoding.UTF8.GetBytes(pars[2])))
                        {
                            XElement root = XElement.Load(ms);
                            field.FieldSchema.FromXmlElement(root);
                        }
                    }
                    // 不触发值改变消息
                    field._value = this.GetControlObjectFieldValue(connection, layer, info.NodeNum, field.FieldIdentifier);

                    obj.Fields.Add(field);
                }

                list.Add(obj);
            }

            return list;
        }

        /// <summary>
        /// 获取控制对象ID集合
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="layer">场景图层</param>
        /// <returns>控制对象ID集合</returns>
        public List<string> GetControlObjectIds(TVPConnectionModel connection, VizSceneLayerEnum layer)
        {
            string cmd = $"{layer}*TREE SEARCH_FOR_CONTAINER_WITH_PROPERTY BUILT_IN*FUNCTION BUILT_IN*FUNCTION*ControlObject";
            string result = connection.EndpointManager.Request(cmd);
            string[] source = result.Trim().Split(' ');

            return source.ToList();
        }

        /// <summary>
        /// 获取控制对象参数
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="layer">场景图层</param>
        /// <param name="nodeNum">场景树编号</param>
        /// <param name="parameter">参数</param>
        /// <returns>描述</returns>
        public string GetControlObjectParameter(TVPConnectionModel connection, VizSceneLayerEnum layer, string nodeNum, VizControlObjectParameters parameter)
        {
            string cmd = $"{layer}*TREE*#{nodeNum}*FUNCTION*ControlObject*{parameter} GET";
            string result = connection.EndpointManager.Request(cmd);

            return result;
        }

        /// <summary>
        /// 获取控制对象返回值
        /// </summary>
        /// <remarks>
        /// 返回结果按照“:”进行分割
        /// FieldIdentifier:VizID_SetGetfunction:Type:Min:Max:MaxChar:Description:AuxField:
        /// </remarks>
        /// <param name="connection">连接</param>
        /// <param name="layer">场景图层</param>
        /// <param name="nodeNum">场景树编号</param>
        /// <returns>控制对象返回值</returns>
        public string GetControlObjectResult(TVPConnectionModel connection, VizSceneLayerEnum layer, string nodeNum)
        {
            connection.EndpointManager.Send($"{layer}*TREE*#{nodeNum}*FUNCTION*ControlObject*in SET LIST");
            string result = connection.EndpointManager.Request($"{layer}*TREE*#{nodeNum}*FUNCTION*ControlObject*result GET");

            return result;
        }

        /// <summary>
        /// 获取控制对象字段值
        /// </summary>
        /// <param name="connection"></param>
        /// <param name="layer"></param>
        /// <param name="nodeNum">场景树编号</param>
        /// <param name="fieldIdentifier">字段</param>
        /// <returns>字段值</returns>
        public string GetControlObjectFieldValue(TVPConnectionModel connection, VizSceneLayerEnum layer, string nodeNum, string fieldIdentifier)
        {
            connection.EndpointManager.Send($"{layer}*TREE*#{nodeNum}*FUNCTION*ControlObject*in SET ON {fieldIdentifier} GET");
            string result = connection.EndpointManager.Request($"{layer}*TREE*#{nodeNum}*FUNCTION*ControlObject*result GET");

            return result;
        }

        /// <summary>
        /// 设置控制对象值
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="layer">场景图层</param>
        /// <param name="treeNodePath">场景节点路径</param>
        /// <param name="fieldIdentifier">字段</param>
        /// <param name="value">值</param>
        public void SetControlObjectValue(TVPConnectionModel connection, VizSceneLayerEnum layer, string treeNodePath, string fieldIdentifier, string value)
        {
            connection.EndpointManager.Send($"{layer}*TREE*{treeNodePath}*FUNCTION*ControlObject*in SET ON {fieldIdentifier} SET {value}");
        }

        /// <summary>
        /// 设置控制对象值
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="layer">场景图层</param>
        /// <param name="obj">控制对象</param>
        public void SetControlObjectValue(TVPConnectionModel connection, VizSceneLayerEnum layer, VizControlObjectModel obj)
        {
            StringBuilder sb = new StringBuilder();
            sb.Append($"{layer}*TREE*{obj.TreeNodePath}*FUNCTION*ControlObject*in SET ON ");
            foreach (VizControlObjectFieldModel field in obj.Fields)
            {
                sb.Append($"{field.FieldIdentifier} SET {field.Value}\\0");
            }

            connection.EndpointManager.Send(sb.ToString());
        }

        /// <summary>
        /// 设置列表控制对象的值
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="layer">场景图层</param>
        /// <param name="listName">列表名称</param>
        /// <param name="listLine">列表值位序</param>
        /// <param name="fieldIdentifier">字段</param>
        /// <param name="value">值</param>
        public void SetxControlObjectListValue(TVPConnectionModel connection, VizSceneLayerEnum layer, string listName, int listLine, string fieldIdentifier, string value)
        {
            connection.EndpointManager.Send($"{layer}*TREE*$object*FUNCTION*ControlObject*in SET WITH {listName} INDEX {listLine} ON {fieldIdentifier} SET {value}");
        }

        /// <summary>
        /// 获取控制对象字段类型
        /// </summary>
        /// <param name="type">字段类型</param>
        /// <returns>控制对象字段类型</returns>
        public VizControlObjectFieldType GetControlObjectFieldType(string type)
        {
            if (type == "text")
                return VizControlObjectFieldType.text;

            if (type == "bool")
                return VizControlObjectFieldType.boolean;

            if (type == "image")
                return VizControlObjectFieldType.image;

            if (type == "richtext")
                return VizControlObjectFieldType.richtext;

            if (type == "triplet")
                return VizControlObjectFieldType.triplet;

            if (type.StartsWith("<?xml"))
                return VizControlObjectFieldType.list;

            return VizControlObjectFieldType.none;
        }
    }
}
