﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Dynamic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using VIZ.Framework.Core;
using VIZ.Framework.Plugin;
using VIZ.TVP.Domain;
using VIZ.TVP.Service;
using VIZ.TVP.Storage;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// Control List视图模型
    /// </summary>
    public class ControlListViewModel : PluginViewModelBase
    {
        public ControlListViewModel()
        {
            // 初始化命令
            this.initCommand();

            // 初始化消息
            this.initMessage();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void initCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void initMessage()
        {
            ApplicationDomainEx.MessageManager.Register<ProgramListItemChangedMessage>(this, this.OnProgramListItemChangedMessage);
            ApplicationDomainEx.MessageManager.Register<ProgramListItemInitedMessage>(this, this.OnProgramListItemInitedMessage);
            ApplicationDomainEx.MessageManager.Register<VizControlObjectFieldValueChangedMessage>(this, this.OnVizControlObjectFieldValueChangedMessage);
        }

        // ==================================================================================
        // Service & Control
        // ==================================================================================

        /// <summary>
        /// ControlObject命令服务
        /// </summary>
        private IVizCommandControlObjectService vizCommandControlObjectService = new VizCommandControlObjectService();

        // ==================================================================================
        // Property
        // ==================================================================================

        #region IsLoading -- 是否正在加载

        private bool isLoading;
        /// <summary>
        /// 是否正在加载
        /// </summary>
        public bool IsLoading
        {
            get { return isLoading; }
            set { isLoading = value; this.RaisePropertyChanged(nameof(IsLoading)); }
        }

        #endregion

        #region ControlObjectList -- 控制对象列表

        private ObservableCollection<VizControlObjectModel> controlObjectList;
        /// <summary>
        /// 控制对象列表
        /// </summary>
        public ObservableCollection<VizControlObjectModel> ControlObjectList
        {
            get { return controlObjectList; }
            set { controlObjectList = value; this.RaisePropertyChanged(nameof(ControlObjectList)); }
        }

        #endregion

        #region SelectedControlObject -- 当前选中的控制对象

        private VizControlObjectModel selectedControlObject;
        /// <summary>
        /// 当前选中的控制对象
        /// </summary>
        public VizControlObjectModel SelectedControlObject
        {
            get { return selectedControlObject; }
            set { selectedControlObject = value; this.RaisePropertyChanged(nameof(SelectedControlObject)); }
        }

        #endregion

        #region SelectedControlObjectField -- 当前选中的控制对象字段

        private VizControlObjectFieldModel selectedControlObjectField;
        /// <summary>
        /// 当前选中的控制对象字段
        /// </summary>
        public VizControlObjectFieldModel SelectedControlObjectField
        {
            get { return selectedControlObjectField; }
            set
            {
                selectedControlObjectField = value;
                this.RaisePropertyChanged(nameof(SelectedControlObjectField));
                this.UpdateSlectedControlObjectField();
            }
        }

        /// <summary>
        /// 更新当前选中的控制对象字段
        /// </summary>
        private void UpdateSlectedControlObjectField()
        {
            VizControlObjectFieldModel field = this.SelectedControlObjectField;
            if (field == null || field.Type != Storage.VizControlObjectFieldType.list)
            {
                this.ListValueColumns = null;
                // 注销属性改变事件
                if (this.ListValue != null)
                {
                    foreach (INotifyPropertyChanged item in this.ListValue)
                    {
                        item.PropertyChanged -= Item_PropertyChanged;
                    }
                }

                this.ListValue = null;
                return;
            }

            if (!field.IsXmlValueLoaded)
            {
                List<GridColumnDefinition> columns;
                List<ExpandoObject> items;
                vizCommandControlObjectService.GetControlObjectXmlData(field.Value, out columns, out items);
                field.ListValueColumns = columns;
                field.ListValue = items;
            }

            // 设置模板
            ControlListView view = this.GetView<ControlListView>();
            foreach (var node in field.FieldSchema.Schema.Fields)
            {
                GridColumnDefinition column = field.ListValueColumns.FirstOrDefault(p => p.FieldName == node.Name);
                if (column == null)
                    continue;

                VizControlObjectFieldType columnType = this.vizCommandControlObjectService.GetControlObjectFieldType(node.Type);

                switch (columnType)
                {
                    case VizControlObjectFieldType.none:
                        break;
                    case VizControlObjectFieldType.text:
                        column.CellTemplate = view.FindResource("ColumnTemplate_TextEdit") as DataTemplate;
                        break;
                    case VizControlObjectFieldType.boolean:
                        column.CellTemplate = view.FindResource("ColumnTemplate_CheckEdit") as DataTemplate;
                        column.Width = new DevExpress.Xpf.Grid.GridColumnWidth(80);
                        break;
                    case VizControlObjectFieldType.richtext:
                        column.CellTemplate = view.FindResource("ColumnTemplate_TextEdit") as DataTemplate;
                        break;
                    case VizControlObjectFieldType.image:
                        column.CellTemplate = view.FindResource("ColumnTemplate_TextEdit") as DataTemplate;
                        break;
                    case VizControlObjectFieldType.list:
                        break;
                }
            }

            // 绑定属性
            foreach (INotifyPropertyChanged item in field.ListValue)
            {
                item.PropertyChanged -= Item_PropertyChanged;
                item.PropertyChanged += Item_PropertyChanged;
            }

            this.ListValueColumns = field.ListValueColumns;
            this.ListValue = field.ListValue;
        }

        /// <summary>
        /// 动态数据项属性改变时触发
        /// </summary>
        private void Item_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (this.SelectedControlObjectField.Type != VizControlObjectFieldType.list)
                return;

            this.SelectedControlObjectField.Value = this.vizCommandControlObjectService.GetControlObjectXml(this.ListValue);
        }

        #endregion

        #region ListValue -- List类型数据列表

        private List<ExpandoObject> listValue;
        /// <summary>
        /// List类型数据列表
        /// </summary>
        public List<ExpandoObject> ListValue
        {
            get { return listValue; }
            set { listValue = value; this.RaisePropertyChanged(nameof(ListValue)); }
        }

        #endregion

        #region ListValueColumns -- List类型数据列信息

        private List<GridColumnDefinition> listValueColumns;
        /// <summary>
        /// List类型数据列信息
        /// </summary>
        public List<GridColumnDefinition> ListValueColumns
        {
            get { return listValueColumns; }
            set { listValueColumns = value; this.RaisePropertyChanged(nameof(ListValueColumns)); }
        }

        #endregion

        // ==================================================================================
        // Command
        // ==================================================================================

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {

        }

        #endregion

        // ==================================================================================
        // Message
        // ==================================================================================

        /// <summary>
        /// 节目单项目改变消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProgramListItemChangedMessage(ProgramListItemChangedMessage msg)
        {
            if (msg.ProgramListItemModel == null)
            {
                this.ControlObjectList = null;
                this.SelectedControlObject = null;
                return;
            }

            this.ControlObjectList = msg.ProgramListItemModel.ControlObjectList;
            this.SelectedControlObject = this.ControlObjectList.FirstOrDefault();
        }

        /// <summary>
        /// 处理节目单项初始化完成消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProgramListItemInitedMessage(ProgramListItemInitedMessage msg)
        {
            // 如果已经加载过控制对象，那么不再加载
            if (msg.ProgramListItemModel.IsLoadedControlObjectList)
                return;

            this.IsLoading = true;

            // 否则加载控制对象列表
            ThreadHelper.SafeRun(() =>
            {
                // 获取
                var result = this.vizCommandControlObjectService.GetControlObjectList(ApplicationDomainEx.TVPConnectionManager.LocalConnection, msg.ProgramListItemModel, Storage.VizSceneLayerEnum.MAIN_SCENE);

                WPFHelper.BeginInvoke(() =>
                {
                    msg.ProgramListItemModel.ControlObjectList.Clear();

                    foreach (var item in result)
                    {
                        msg.ProgramListItemModel.ControlObjectList.Add(item);
                    }

                    msg.ProgramListItemModel.IsLoadedControlObjectList = true;

                    this.IsLoading = false;
                });
            });
        }

        /// <summary>
        /// 控制对象值改变消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnVizControlObjectFieldValueChangedMessage(VizControlObjectFieldValueChangedMessage msg)
        {
            if (msg.FieldModel.ProgramListItemModel != ApplicationDomainEx.SelectionManager.ProgramListItemModel)
                return;

            TVPConnectionModel connection = ApplicationDomainEx.TVPConnectionManager.LocalConnection;
            string treeNodePath = msg.FieldModel.VizControlObjectModel.TreeNodeInfo.NodeFullPath;
            this.vizCommandControlObjectService.SetControlObjectValue(connection, Storage.VizSceneLayerEnum.MAIN_SCENE, treeNodePath, msg.FieldModel.FieldIdentifier, msg.FieldModel.Value);
        }

        // ==================================================================================
        // Public Function
        // ==================================================================================

        /// <summary>
        /// 销毁
        /// </summary>
        public override void Dispose()
        {

        }

    }
}
