﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.Framework.Plugin;
using VIZ.TVP.Connection;
using VIZ.TVP.Domain;
using VIZ.TVP.Service;
using VIZ.TVP.Storage;
using static VIZ.Framework.Core.WPFHelper;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// VIZ渲染视图模型
    /// </summary>
    public class VizRenderViewModel : PluginViewModelBase, IVizSupport
    {
        public VizRenderViewModel()
        {
            // 初始化控制器
            this.initController();

            // 初始化命令
            this.initCommand();

            // 初始化消息
            this.initMessage();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void initCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
            this.WindowHostSizeChangedCommand = new VCommand<System.Windows.SizeChangedEventArgs>(this.WindowHostSizeChanged);
            this.PlayCommand = new VCommand(this.Play);
            this.ContinueCommand = new VCommand(this.Continue);
            this.StopCommand = new VCommand(this.Stop);
            this.ShowTSCommand = new VCommand(this.ShowTS);
            this.HideTSCommand = new VCommand(this.HideTS);
            this.ShowSACommand = new VCommand(this.ShowSA);
            this.HideSACommand = new VCommand(this.HideSA);
            this.ShowBBCommand = new VCommand(this.ShowBB);
            this.HideBBCommand = new VCommand(this.HideBB);
            this.ShowRGBCommand = new VCommand(this.ShowRGB);
            this.ShowKeyCommand = new VCommand(this.ShowKey);
            this.ShowKeyPreviewCommand = new VCommand(this.ShowKeyPreview);
        }

        /// <summary>
        /// 初始化控制器
        /// </summary>
        private void initController()
        {
            this.vizController = new VizController(this);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void initMessage()
        {
            ApplicationDomainEx.MessageManager.Register<ProgramListItemChangedMessage>(this, this.OnProgramListItemChangedMessage);
        }

        // ================================================================================
        // Property
        // ================================================================================

        #region IsEngineReady -- 引擎是否准备完毕

        private bool isEngineReady;
        /// <summary>
        /// 引擎是否准备完毕
        /// </summary>
        public bool IsEngineReady
        {
            get { return isEngineReady; }
            set { isEngineReady = value; this.RaisePropertyChanged(nameof(IsEngineReady)); }
        }

        #endregion

        // ================================================================================
        // Controller & Service
        // ================================================================================

        /// <summary>
        /// Viz控制器
        /// </summary>
        private VizController vizController;

        /// <summary>
        /// VIZ命令服务
        /// </summary>
        private IVizCommandService vizCommandService = new VizCommandService();

        // ================================================================================
        // Command
        // ================================================================================

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {
            if (this.IsAlreadyLoaded)
                return;

            VizRenderView view = this.GetView<VizRenderView>();
            if (view == null)
                return;

            ThreadHelper.SafeRun(() =>
            {
                // 等待能正确获取到界面宽度时再嵌入VIZ引擎
                double width = 0;
                while (width == 0)
                {
                    WPFHelper.Invoke(() =>
                    {
                        width = view.ActualWidth;
                    });

                    System.Threading.Thread.Sleep(500);
                }

                // 启动引擎
                this.StartVizEngine();
            });

            this.IsAlreadyLoaded = true;
        }

        #endregion

        #region WindowHostSizeChangedCommand -- 宿主容器大小改变命令

        /// <summary>
        /// 宿主容器大小改变命令
        /// </summary>
        public VCommand<System.Windows.SizeChangedEventArgs> WindowHostSizeChangedCommand { get; set; }

        /// <summary>
        /// 宿主容器大小改变
        /// </summary>
        private void WindowHostSizeChanged(System.Windows.SizeChangedEventArgs e)
        {
            LocalInfoEntity info = ApplicationDomainEx.DataBaseManager.LocalInfo;
            TVPConnectionModel local = ApplicationDomainEx.TVPConnectionManager.LocalConnection;

            if (local == null || !local.IsConnected)
                return;

            var dpi = WPFHelper.GetDpiByGraphics();

            int width = (int)(e.NewSize.Width * (dpi.X / 96d));
            int height = (int)(e.NewSize.Height * (dpi.Y / 96d));

            ImageHelper.AdjustSize(width, height, 1920, 1080, out width, out height);

            if (info.LocalEngineType == LocalEngineType.Eng3)
            {
                local.EndpointManager.Send($"RENDERER WINDOW_RESIZE {width} {height}");
            }
            else
            {
                local.EndpointManager.Send($"{ApplicationDomainEx.LocalVizManager.VizRendererName} WINDOW_RESIZE {width} {height}");
            }
        }

        #endregion

        #region PlayCommand -- 播放命令

        /// <summary>
        /// 播放命令
        /// </summary>
        public VCommand PlayCommand { get; set; }

        /// <summary>
        /// 播放
        /// </summary>
        private void Play()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;
            ProgramListItemModel item = ApplicationDomainEx.SelectionManager.ProgramListItemModel;

            this.vizCommandService.Start(conn, null, VizLayerEnum.MAIN_LAYER);
        }

        #endregion

        #region ContinueCommand -- 继续命令

        /// <summary>
        /// 继续命令
        /// </summary>
        public VCommand ContinueCommand { get; set; }

        /// <summary>
        /// 继续
        /// </summary>
        private void Continue()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;
            ProgramListItemModel item = ApplicationDomainEx.SelectionManager.ProgramListItemModel;

            this.vizCommandService.Continue(conn, null, VizLayerEnum.MAIN_LAYER);
        }

        #endregion

        #region StopCommand -- 停止命令

        /// <summary>
        /// 停止命令
        /// </summary>
        public VCommand StopCommand { get; set; }

        /// <summary>
        /// 停止
        /// </summary>
        private void Stop()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;
            ProgramListItemModel item = ApplicationDomainEx.SelectionManager.ProgramListItemModel;

            this.vizCommandService.Stop(conn, null, VizLayerEnum.MAIN_LAYER);
        }

        #endregion

        // --------------------------------
        // 安全框

        #region ShowTSCommand -- 显示安全框命令

        /// <summary>
        /// 显示安全框命令
        /// </summary>
        public VCommand ShowTSCommand { get; set; }

        /// <summary>
        /// 显示安全框
        /// </summary>
        private void ShowTS()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;

            this.vizCommandService.ShowTS(conn, true);
        }

        #endregion

        #region HideTSCommand -- 隐藏安全框命令

        /// <summary>
        /// 隐藏安全框命令
        /// </summary>
        public VCommand HideTSCommand { get; set; }

        /// <summary>
        /// 隐藏安全框
        /// </summary>
        private void HideTS()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;

            this.vizCommandService.ShowTS(conn, false);
        }

        #endregion

        // --------------------------------
        // 安全区域

        #region ShowSACommand -- 显示安全区域命令

        /// <summary>
        /// 显示安全区域命令
        /// </summary>
        public VCommand ShowSACommand { get; set; }

        /// <summary>
        /// 显示安全区域
        /// </summary>
        private void ShowSA()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;

            this.vizCommandService.ShowSA(conn, true);
        }

        #endregion

        #region HideSACommand -- 隐藏安全区域命令

        /// <summary>
        /// 隐藏安全区域命令
        /// </summary>
        public VCommand HideSACommand { get; set; }

        /// <summary>
        /// 隐藏安全区域
        /// </summary>
        private void HideSA()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;

            this.vizCommandService.ShowSA(conn, false);
        }

        #endregion

        // --------------------------------
        // 边界框

        #region ShowBBCommand -- 显示边界框命令

        /// <summary>
        /// 显示边界框命令
        /// </summary>
        public VCommand ShowBBCommand { get; set; }

        /// <summary>
        /// 显示边界框
        /// </summary>
        private void ShowBB()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;

            this.vizCommandService.ShowBB(conn, true);
        }

        #endregion

        #region HideBBCommand -- 隐藏边界框命令

        /// <summary>
        /// 隐藏边界框命令
        /// </summary>
        public VCommand HideBBCommand { get; set; }

        /// <summary>
        /// 隐藏边界框
        /// </summary>
        private void HideBB()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;

            this.vizCommandService.ShowBB(conn, false);
        }

        #endregion

        // --------------------------------
        // 其他

        #region ShowRGBCommand -- 显示RGB命令

        /// <summary>
        /// 显示RGB命令
        /// </summary>
        public VCommand ShowRGBCommand { get; set; }

        /// <summary>
        /// 显示RGB
        /// </summary>
        private void ShowRGB()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;

            this.vizCommandService.ShowRGB(conn);
        }

        #endregion

        #region ShowKeyCommand -- 显示键命令

        /// <summary>
        /// 显示键命令
        /// </summary>
        public VCommand ShowKeyCommand { get; set; }

        /// <summary>
        /// 显示键
        /// </summary>
        private void ShowKey()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;

            this.vizCommandService.ShowKey(conn);
        }

        #endregion

        #region ShowKeyPreviewCommand -- 显示预览键命令

        /// <summary>
        /// 显示预览键命令
        /// </summary>
        public VCommand ShowKeyPreviewCommand { get; set; }

        /// <summary>
        /// 显示预览键
        /// </summary>
        private void ShowKeyPreview()
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;

            this.vizCommandService.ShowKeyPreview(conn);
        }

        #endregion

        // ================================================================================
        // Message
        // ================================================================================

        /// <summary>
        /// 节目当项目切换
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProgramListItemChangedMessage(ProgramListItemChangedMessage msg)
        {
            // 引擎未准备好 || 当前没有选中的节目单项目 则不处理
            if (!this.IsEngineReady || ApplicationDomainEx.SelectionManager.ProgramListItemModel == null)
                return;

            TVPConnectionModel conn = ApplicationDomainEx.TVPConnectionManager.LocalConnection;

            this.vizCommandService.SetObject(conn, msg.ProgramListItemModel.Path, VizLayerEnum.MAIN_LAYER);

            // 发送节目单项初始化完成消息
            WPFHelper.BeginInvoke(() =>
            {
                ProgramListItemInitedMessage message = new ProgramListItemInitedMessage();
                message.ProgramListItemModel = msg.ProgramListItemModel;

                ApplicationDomainEx.MessageManager.Send(message);
            });
        }

        // ================================================================================
        // Public Function
        // ================================================================================

        /// <summary>
        /// 启动VIZ引擎
        /// </summary>
        public void StartVizEngine()
        {
            VizRenderView view = this.GetView<VizRenderView>();
            if (view == null)
                return;

            LocalInfoEntity info = ApplicationDomainEx.DataBaseManager.LocalInfo;
            TVPConnectionModel local = ApplicationDomainEx.TVPConnectionManager.LocalConnection;
            if (local != null)
            {
                local.Dispose();
            }

            local = new TVPConnectionModel(new TVPConnectionEntity());
            local.InitEndpointManager(new VizEndpointManager("local", info.VIZ_IP, info.VIZ_Port));

            ApplicationDomainEx.TVPConnectionManager.LocalConnection = local;

            string path = null;

            switch (info.LocalEngineType)
            {
                case LocalEngineType.Eng3: path = info.VIZ_Eng3Path; break;
                case LocalEngineType.Eng4: path = info.VIZ_Eng4Path; break;
            }

            this.vizController.StartVizEngine(view, path, local);
        }

        /// <summary>
        /// 销毁
        /// </summary>
        public override void Dispose()
        {

        }
    }
}
