﻿using log4net;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Plugin;
using VIZ.TVP.Domain;
using VIZ.TVP.Plugin;
using VIZ.TVP.Storage;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// 插件控制器
    /// </summary>
    /// <remarks>
    /// 1. 插件命名空间为 VIZ.TVP.Plugin 开头
    /// 2. 实现IPluginAssemblyLifeCycle接口
    /// </remarks>
    public class PluginController
    {
        /// <summary>
        /// 日志
        /// </summary>
        private static readonly ILog log = LogManager.GetLogger(typeof(PluginController));

        /// <summary>
        /// 插件程序集前缀
        /// </summary>
        public const string PLUGIN_ASSEMBLY_PREFIX = "VIZ.TVP.";

        /// <summary>
        /// 插件控制器
        /// </summary>
        /// <param name="support">支持</param>
        public PluginController(IPluginSupport support)
        {
            this.Support = support;
        }

        /// <summary>
        /// 支持
        /// </summary>
        public IPluginSupport Support { get; private set; }

        /// <summary>
        /// 加载插件配置
        /// </summary>
        public void LoadPluginConfig()
        {
            Dictionary<string, PluginGroupModel> groupModels = new Dictionary<string, PluginGroupModel>();

            foreach (var item in ApplicationDomainEx.DataBaseManager.CsvContext.PluginConfig)
            {
                if (!groupModels.TryGetValue(item.Group, out PluginGroupModel groupModel))
                {
                    groupModel = new PluginGroupModel();
                    groupModel.Group = item.Group;

                    groupModels.Add(groupModel.Group, groupModel);
                }

                PluginConfigModel model = new PluginConfigModel();
                model.PropertyFromConfig(item);

                groupModel.Items.Add(model);
            }

            this.Support.PluginGroups = groupModels.Values.ToList();

            LocalInfoEntity info = ApplicationDomainEx.DataBaseManager.LocalInfo;

            if (string.IsNullOrWhiteSpace(info.PluginGroup))
            {
                this.Support.SelectedPluginGroup = this.Support.PluginGroups.FirstOrDefault();
            }
            else
            {
                this.Support.SelectedPluginGroup = this.Support.PluginGroups.FirstOrDefault(p => p.Group == info.PluginGroup) ?? this.Support.PluginGroups.FirstOrDefault();
            }
        }

        /// <summary>
        /// 加载插件
        /// </summary>
        public void LoadPlugins()
        {
            // 加载插件
            this.executeLoadPlugins();

            if (this.Support.SelectedPluginGroup != null)
            {
                // 加载外置插件
                this.executeLoadFilePlugins();
            }
        }

        /// <summary>
        /// 执行加载内置插件
        /// </summary>
        private void executeLoadPlugins()
        {
            Assembly assembly = this.GetType().Assembly;
            string version = assembly.GetCustomAttribute<AssemblyVersionAttribute>()?.Version;
            Type[] types = assembly.GetTypes();

            Type lifeCycleType = typeof(IPluginLifeCycle);

            foreach (Type type in types)
            {
                if (!type.IsClass || !lifeCycleType.IsAssignableFrom(type))
                    continue;

                IPluginLifeCycle lifeCycle = type.Assembly.CreateInstance(type.FullName) as IPluginLifeCycle;
                if (lifeCycle == null)
                {
                    log.Error($"init plugin type: {type.FullName} error.");
                    continue;
                }

                PluginInfo info = lifeCycle.Register();
                info.LifeCycle = lifeCycle;
                info.Version = version;
                lifeCycle.Initialize();
                ApplicationDomainEx.PluginManager.Plugins.Add(info);
            }
        }

        /// <summary>
        /// 执行加载外置插件
        /// </summary>
        private void executeLoadFilePlugins()
        {
            string[] files = System.IO.Directory.GetFiles(AppDomain.CurrentDomain.BaseDirectory);

            Type templateLifeCycleType = typeof(ITemplatePluginLifeCycle);
            Type pluginLifeCycleType = typeof(IPluginLifeCycle);

            foreach (string file in files)
            {
                string fileName = System.IO.Path.GetFileName(file);

                if (!fileName.StartsWith(PLUGIN_ASSEMBLY_PREFIX) || !fileName.EndsWith(".dll"))
                    continue;

                Assembly assembly = Assembly.LoadFile(file);
                string version = assembly.GetCustomAttribute<AssemblyVersionAttribute>()?.Version;
                Type[] types = assembly.GetTypes();

                foreach (Type type in types)
                {
                    if (!type.IsClass)
                        continue;

                    // 模板插件
                    if (templateLifeCycleType.IsAssignableFrom(type))
                    {
                        ITemplatePluginLifeCycle lifeCycle = type.Assembly.CreateInstance(type.FullName) as ITemplatePluginLifeCycle;
                        if (lifeCycle == null)
                        {
                            log.Error($"init template plugin type: {type.FullName} error.");
                            continue;
                        }

                        PluginInfo info = lifeCycle.Register();
                        if (!this.Support.SelectedPluginGroup.Items.Any(p => p.PluginID == info.ID && p.Type == PluginTypeEnum.TemplatePlugin))
                        {
                            continue;
                        }

                        info.LifeCycle = lifeCycle;
                        info.Version = version;
                        lifeCycle.Initialize();
                        ApplicationDomainEx.PluginManager.TemplatePlugins.Add(info);

                        continue;
                    }

                    // 插件
                    if (pluginLifeCycleType.IsAssignableFrom(type))
                    {
                        IPluginLifeCycle lifeCycle = type.Assembly.CreateInstance(type.FullName) as IPluginLifeCycle;
                        if (lifeCycle == null)
                        {
                            log.Error($"init plugin type: {type.FullName} error.");
                            continue;
                        }

                        PluginInfo info = lifeCycle.Register();
                        if (!this.Support.SelectedPluginGroup.Items.Any(p => p.PluginID == info.ID && p.Type == PluginTypeEnum.Plugin))
                        {
                            continue;
                        }

                        info.LifeCycle = lifeCycle;
                        info.Version = version;
                        lifeCycle.Initialize();
                        ApplicationDomainEx.PluginManager.Plugins.Add(info);

                        continue;
                    }
                }
            }
        }
    }
}
