﻿using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Configuration;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Common;
using VIZ.Framework.Plugin;
using VIZ.TVP.Domain;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// 插件视图模型
    /// </summary>
    public class PluginPanelViewModel : PluginViewModelBase
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(PluginPanelViewModel));

        /// <summary>
        /// 插件视图模型
        /// </summary>
        public PluginPanelViewModel()
        {
            // 初始化消息
            this.initMessage();
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void initMessage()
        {
            ApplicationDomainEx.MessageManager.Register<ProgramListItemChangedMessage>(this, this.OnProgramListItemChangedMessage);
            ApplicationDomainEx.MessageManager.Register<ProgramListDeleteMessage>(this, this.OnProgramListDeleteMessage);
            ApplicationDomainEx.MessageManager.Register<ProgramListItemDeleteMessage>(this, this.OnProgramListItemDeleteMessage);
        }

        // ==========================================================================
        // Property
        // ==========================================================================

        #region ItemsSource -- 插件项集合

        private ObservableCollection<NavigationConfig> itemsSource = new ObservableCollection<NavigationConfig>();
        /// <summary>
        /// 插件项集合
        /// </summary>
        public ObservableCollection<NavigationConfig> ItemsSource
        {
            get { return itemsSource; }
            set { itemsSource = value; this.RaisePropertyChanged(nameof(ItemsSource)); }
        }

        #endregion

        #region SelectedItem -- 当前选中项

        private NavigationConfig selectedItem;
        /// <summary>
        /// 当前选中项
        /// </summary>
        public NavigationConfig SelectedItem
        {
            get { return selectedItem; }
            set { selectedItem = value; this.RaisePropertyChanged(nameof(SelectedItem)); }
        }

        #endregion


        // ==========================================================================
        // Message
        // ==========================================================================

        /// <summary>
        /// 当切换节目单项时触发
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProgramListItemChangedMessage(ProgramListItemChangedMessage msg)
        {
            lock (this.ItemsSource)
            {
                // 该节目单项没有插件配置
                if (msg.ProgramListItemModel == null || string.IsNullOrWhiteSpace(msg.ProgramListItemModel.PluginID))
                {
                    this.SelectedItem = null;
                    return;
                }

                // 该节目单项对应的插件已经被实例化过
                NavigationConfig config = this.ItemsSource.FirstOrDefault(p => p.Key == msg.ProgramListItemModel.ObjectID);
                if (config != null)
                {
                    this.SelectedItem = config;
                    return;
                }

                // 为该节目单项实例化插件
                PluginInfo pluginInfo = ApplicationDomainEx.PluginManager.TemplatePlugins.FirstOrDefault(p => p.ID == msg.ProgramListItemModel.PluginID);
                if (pluginInfo == null)
                {
                    log.Error($"can`t found plugin id: {msg.ProgramListItemModel.PluginID}, plugin is unloaded or not exists");
                    return;
                }

                config = new NavigationConfig();
                config.Key = msg.ProgramListItemModel.ObjectID;
                config.ViewType = pluginInfo.ViewInfo.ViewType;
                config.ViewModelType = pluginInfo.ViewInfo.ViewModelType;

                this.ItemsSource.Add(config);
                this.SelectedItem = config;
            }
        }

        /// <summary>
        /// 当删除节目单时触发
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProgramListDeleteMessage(ProgramListDeleteMessage msg)
        {
            lock (this.itemsSource)
            {
                if (msg.DeletedProgramListModel == null)
                    return;

                foreach (var item in msg.DeletedProgramListModel.Items)
                {
                    // 未配置插件
                    if (string.IsNullOrWhiteSpace(item.PluginID))
                        continue;

                    // 未找到相关插件视图
                    NavigationConfig config = this.ItemsSource.FirstOrDefault(p => p.Key == item.ObjectID);

                    // 尝试移除导航配置
                    this.tryRemoveItem(config);
                }
            }
        }

        /// <summary>
        /// 当删除节目单项时触发
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProgramListItemDeleteMessage(ProgramListItemDeleteMessage msg)
        {
            lock (this.ItemsSource)
            {
                if (msg.DeletedProgramListItemModel == null)
                    return;

                // 未配置插件
                if (string.IsNullOrWhiteSpace(msg.DeletedProgramListItemModel.PluginID))
                    return;

                // 未找到相关插件视图
                NavigationConfig config = this.ItemsSource.FirstOrDefault(p => p.Key == msg.DeletedProgramListItemModel.ObjectID);

                // 尝试移除导航配置
                this.tryRemoveItem(config);
            }
        }

        // ==========================================================================
        // Public Function
        // ==========================================================================

        /// <summary>
        /// 销毁
        /// </summary>
        public override void Dispose()
        {

        }

        // ==========================================================================
        // Private Function
        // ==========================================================================

        /// <summary>
        /// 尝试移除导航配置
        /// </summary>
        /// <param name="config">配置</param>
        private void tryRemoveItem(NavigationConfig config)
        {
            if (config == null)
                return;

            this.ItemsSource.Remove(config);
            if (this.SelectedItem == config)
            {
                this.SelectedItem = null;
            }

            if (config.View != null && config.View.TryGetTarget(out object view))
            {
                try
                {
                    IDisposable disposable = view as IDisposable;
                    disposable?.Dispose();
                }
                catch (Exception ex)
                {
                    log.Error(ex);
                }
            }

            if (config.ViewModel != null && config.ViewModel.TryGetTarget(out object vm))
            {
                try
                {
                    IDisposable disposable = vm as IDisposable;
                    disposable?.Dispose();
                }
                catch (Exception ex)
                {
                    log.Error(ex);
                }
            }
        }
    }
}
