﻿using log4net;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Plugin;
using VIZ.TVP.Domain;
using VIZ.TVP.Plugin;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// 插件控制器
    /// </summary>
    /// <remarks>
    /// 1. 插件命名空间为 VIZ.TVP.Plugin 开头
    /// 2. 实现IPluginAssemblyLifeCycle接口
    /// </remarks>
    public class PluginController
    {
        /// <summary>
        /// 日志
        /// </summary>
        private static readonly ILog log = LogManager.GetLogger(typeof(PluginController));

        /// <summary>
        /// 插件程序集前缀
        /// </summary>
        public const string PLUGIN_ASSEMBLY_PREFIX = "VIZ.TVP.Plugin.";

        /// <summary>
        /// 插件控制器
        /// </summary>
        /// <param name="support">支持</param>
        public PluginController(IPluginSupport support)
        {
            this.Support = support;
        }

        /// <summary>
        /// 支持
        /// </summary>
        public IPluginSupport Support { get; private set; }

        /// <summary>
        /// 加载插件
        /// </summary>
        public void LoadPlugins()
        {
            // 加载插件
            this.executeLoadPlugins();

            // 加载模板插件
            this.executeLoadTemplatePlugins();
        }

        /// <summary>
        /// 执行加载内置插件
        /// </summary>
        private void executeLoadPlugins()
        {
            Assembly assembly = this.GetType().Assembly;
            string version = assembly.GetCustomAttribute<AssemblyVersionAttribute>()?.Version;
            Type[] types = assembly.GetTypes();

            Type lifeCycleType = typeof(IPluginLifeCycle);

            foreach (Type type in types)
            {
                if (!type.IsClass || !lifeCycleType.IsAssignableFrom(type))
                    continue;

                IPluginLifeCycle lifeCycle = type.Assembly.CreateInstance(type.FullName) as IPluginLifeCycle;
                if (lifeCycle == null)
                {
                    log.Error($"init plugin type: {type.FullName} error.");
                    continue;
                }

                PluginInfo info = lifeCycle.Register();
                info.LifeCycle = lifeCycle;
                info.Version = version;
                lifeCycle.Initialize();
                ApplicationDomainEx.PluginManager.Plugins.Add(info);
            }
        }

        /// <summary>
        /// 执行加载模板插件
        /// </summary>
        private void executeLoadTemplatePlugins()
        {
            string[] files = System.IO.Directory.GetFiles(AppDomain.CurrentDomain.BaseDirectory);

            foreach (string file in files)
            {
                string fileName = System.IO.Path.GetFileName(file);

                if (!fileName.StartsWith(PLUGIN_ASSEMBLY_PREFIX) || !fileName.EndsWith(".dll"))
                    continue;

                Assembly assembly = Assembly.LoadFile(file);
                string version = assembly.GetCustomAttribute<AssemblyVersionAttribute>()?.Version;
                Type[] types = assembly.GetTypes();

                Type lifeCycleType = typeof(ITemplatePluginLifeCycle);

                foreach (Type type in types)
                {
                    if (!type.IsClass || !lifeCycleType.IsAssignableFrom(type))
                        continue;

                    ITemplatePluginLifeCycle lifeCycle = type.Assembly.CreateInstance(type.FullName) as ITemplatePluginLifeCycle;
                    if (lifeCycle == null)
                    {
                        log.Error($"init template plugin type: {type.FullName} error.");
                        continue;
                    }

                    PluginInfo info = lifeCycle.Register();
                    info.LifeCycle = lifeCycle;
                    info.Version = version;
                    lifeCycle.Initialize();
                    ApplicationDomainEx.PluginManager.TemplatePlugins.Add(info);
                }
            }
        }
    }
}
