﻿using DevExpress.Xpf.Core;
using DevExpress.Xpf.Core.Native;
using DevExpress.Xpf.Editors;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.Framework.Plugin;
using VIZ.TVP.Domain;
using VIZ.TVP.Service;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// 节目单视图模型
    /// </summary>
    public class ProgramListViewModel : PluginViewModelBase, IProgramListViewService
    {
        public ProgramListViewModel()
        {
            // 初始化命令
            this.initCommand();

            // 初始化消息
            this.initMessage();

            // 注册服务
            ApplicationDomainEx.ServiceManager.AddService("PROGRAM_LIST_VIEW_SERVICE", this);
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void initCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
            this.AddProgramListCommand = new VCommand<TabControlTabAddingEventArgs>(this.AddProgramList);
            this.AddGroupCommand = new VCommand(this.AddGroup);
            this.RenameGroupCommand = new VCommand(this.RenameGroup);
            this.DeleteGroupCommand = new VCommand(this.DeleteGroup);
            this.SceneTemplateAddToListCommand = new VCommand(this.SceneTemplateAddToList);
            this.OtherTemplateAddToListCommand = new VCommand(this.OtherTemplateAddToList);
            this.SceneTemplateUpdateCommand = new VCommand(this.SceneTemplateUpdate);
            this.SceneTemplateDeleteCommand = new VCommand(this.SceneTemplateDelete);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void initMessage()
        {
            ApplicationDomainEx.MessageManager.Register<ProjectChangedMessage>(this, this.OnProjectChangedMessage);
            ApplicationDomainEx.MessageManager.Register<ProjectSaveMessage>(this, this.OnProjectSaveMessage);
        }


        // ======================================================================================
        // Field
        // ======================================================================================

        /// <summary>
        /// GH服务
        /// </summary>
        private IGHService ghService = new GHService();

        /// <summary>
        /// 节目单服务
        /// </summary>
        private IProgramService programService = new ProgramService();

        // ======================================================================================
        // Property
        // ======================================================================================

        #region SceneTemplateModels -- 场景模板模型集合

        private ObservableCollection<ProgramTemplateModel> sceneTemplateModels;
        /// <summary>
        /// 场景模板集合
        /// </summary>
        public ObservableCollection<ProgramTemplateModel> SceneTemplateModels
        {
            get { return sceneTemplateModels; }
            set { sceneTemplateModels = value; this.RaisePropertyChanged(nameof(SceneTemplateModels)); }
        }

        #endregion

        #region SelectedSceneTemplateModel -- 当前选中的场景模板模型

        private ProgramTemplateModel selectedSceneTemplateModel;
        /// <summary>
        /// 当前选中的场景模板模型
        /// </summary>
        public ProgramTemplateModel SelectedSceneTemplateModel
        {
            get { return selectedSceneTemplateModel; }
            set { selectedSceneTemplateModel = value; this.RaisePropertyChanged(nameof(SelectedSceneTemplateModel)); }
        }

        #endregion

        #region OtherTemplateModels -- 其他模板模型集合

        private ObservableCollection<ProgramTemplateModel> otherTemplateModels;
        /// <summary>
        /// 其他模板模型集合
        /// </summary>
        public ObservableCollection<ProgramTemplateModel> OtherTemplateModels
        {
            get { return otherTemplateModels; }
            set { otherTemplateModels = value; this.RaisePropertyChanged(nameof(OtherTemplateModels)); }
        }

        #endregion

        #region SelectedOtherTemplateModel -- 当前选中的其他模板模型

        private ProgramTemplateModel selectedOtherTemplateModel;
        /// <summary>
        /// 当前选中的其他模板模型
        /// </summary>
        public ProgramTemplateModel SelectedOtherTemplateModel
        {
            get { return selectedOtherTemplateModel; }
            set { selectedOtherTemplateModel = value; this.RaisePropertyChanged(nameof(SelectedOtherTemplateModel)); }
        }

        #endregion

        #region ProgramListModels -- 节目单集合

        private ObservableCollection<ProgramListModel> programListModels;
        /// <summary>
        /// 节目单集合
        /// </summary>
        public ObservableCollection<ProgramListModel> ProgramListModels
        {
            get { return programListModels; }
            set { programListModels = value; this.RaisePropertyChanged(nameof(ProgramListModels)); }
        }

        #endregion

        #region SelectedProgramListModel -- 当前选中的节目单

        private ProgramListModel selectedProgramListModel;
        /// <summary>
        /// 当前选中的节目单
        /// </summary>
        public ProgramListModel SelectedProgramListModel
        {
            get { return selectedProgramListModel; }
            set { selectedProgramListModel = value; this.RaisePropertyChanged(nameof(SelectedProgramListModel)); }
        }

        #endregion

        #region IsProgramTemplateLoading -- 节目模板是否在加载

        private bool isProgramTemplateLoading;
        /// <summary>
        /// 节目模板是否在加载
        /// </summary>
        public bool IsProgramTemplateLoading
        {
            get { return isProgramTemplateLoading; }
            set { isProgramTemplateLoading = value; this.RaisePropertyChanged(nameof(IsProgramTemplateLoading)); }
        }

        #endregion

        #region IsProgramListLoading -- 节目单是否在加载

        private bool isProgramListLoading;
        /// <summary>
        /// 节目单是否在加载
        /// </summary>
        public bool IsProgramListLoading
        {
            get { return isProgramListLoading; }
            set { isProgramListLoading = value; this.RaisePropertyChanged(nameof(IsProgramListLoading)); }
        }

        #endregion

        // ======================================================================================
        // Command
        // ======================================================================================

        #region LoadedCommand -- 加载完成命令

        /// <summary>
        /// 加载完成命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载完成
        /// </summary>
        private void Loaded()
        {

        }

        #endregion

        #region AddProgramListCommand -- 添加节目单命令

        /// <summary>
        /// 添加节目单命令
        /// </summary>
        public VCommand<TabControlTabAddingEventArgs> AddProgramListCommand { get; set; }

        /// <summary>
        /// 添加节目单
        /// </summary>
        private void AddProgramList(TabControlTabAddingEventArgs e)
        {
            ProgramListModel model = new ProgramListModel(new Storage.ProgramListEntity());
            model.ListID = Guid.NewGuid().ToString();
            model.Name = "新分组";

            e.Item = model;
        }

        #endregion

        #region AddGroupCommand -- 添加分组命令

        /// <summary>
        /// 添加分组命令
        /// </summary>
        public VCommand AddGroupCommand { get; set; }

        /// <summary>
        /// 添加分组
        /// </summary>
        private void AddGroup()
        {
            ProgramListModel model = new ProgramListModel(new Storage.ProgramListEntity());
            model.ListID = Guid.NewGuid().ToString();
            model.Name = "新分组";

            this.ProgramListModels.Add(model);
        }

        #endregion

        #region RenameGroupCommand -- 重命名分组命令

        /// <summary>
        /// 重命名分组命令
        /// </summary>
        public VCommand RenameGroupCommand { get; set; }

        /// <summary>
        /// 重命名分组
        /// </summary>
        private void RenameGroup()
        {
            if (this.SelectedProgramListModel == null)
                return;

            ProgramListNameWindow window = new ProgramListNameWindow();
            window.inputView.Text = this.SelectedProgramListModel.Name;
            window.ShowDialog();

            if (!window.inputView.IsEnter)
                return;

            this.SelectedProgramListModel.Name = window.inputView.Text.Trim();
        }

        #endregion

        #region DeleteGroupCommand -- 删除分组命令

        /// <summary>
        /// 删除分组命令
        /// </summary>
        public VCommand DeleteGroupCommand { get; set; }

        /// <summary>
        /// 删除分组
        /// </summary>
        private void DeleteGroup()
        {
            if (this.SelectedProgramListModel == null)
                return;

            if (DXMessageBox.Show($"确定删除节目单{this.SelectedProgramListModel.Name}?", "提示", System.Windows.MessageBoxButton.OKCancel) != System.Windows.MessageBoxResult.OK)
                return;

            this.ProgramListModels.Remove(this.SelectedProgramListModel);
        }

        #endregion

        #region SceneTemplateAddToListCommand -- 场景模板添加至节目单命令

        /// <summary>
        /// 场景模板添加至节目单命令
        /// </summary>
        public VCommand SceneTemplateAddToListCommand { get; set; }

        /// <summary>
        /// 场景模板添加至节目单
        /// </summary>
        private void SceneTemplateAddToList()
        {
            if (this.SelectedSceneTemplateModel == null)
                return;

            if (this.SelectedProgramListModel == null)
                return;

            ProgramListItemModel model = new ProgramListItemModel(new Storage.ProgramListItemEntity());
            model.TemplateID = this.SelectedSceneTemplateModel.TemplateID;
            model.ListID = this.SelectedProgramListModel.ListID;
            model.ProgramListModel = this.SelectedProgramListModel;
            model.FromProgramTemplateModel(this.SelectedSceneTemplateModel);
            if (model.ThumbnailBitmap == null)
            {
                ThreadHelper.SafeRun(() =>
                {
                    var bitmap = this.ghService.GetImage(model.Thumbnail);

                    WPFHelper.BeginInvoke(() =>
                    {
                        model.ThumbnailBitmap = bitmap;
                    });
                });
            }

            this.SelectedProgramListModel.Items.Add(model);
        }

        #endregion

        #region OtherTemplateAddToListCommand -- 其他模板添加至节目单命令

        /// <summary>
        /// 其他模板添加至节目单命令
        /// </summary>
        public VCommand OtherTemplateAddToListCommand { get; set; }

        /// <summary>
        /// 其他模板添加至节目单
        /// </summary>
        private void OtherTemplateAddToList()
        {
            if (this.SelectedOtherTemplateModel == null)
                return;

            if (this.SelectedProgramListModel == null)
                return;

            ProgramListItemModel model = new ProgramListItemModel(new Storage.ProgramListItemEntity());
            model.TemplateID = this.SelectedOtherTemplateModel.TemplateID;
            model.ListID = this.SelectedProgramListModel.ListID;
            model.ProgramListModel = this.SelectedProgramListModel;
            model.FromProgramTemplateModel(this.SelectedOtherTemplateModel);
            if (model.ThumbnailBitmap == null)
            {
                ThreadHelper.SafeRun(() =>
                {
                    var bitmap = this.ghService.GetImage(model.Thumbnail);

                    WPFHelper.BeginInvoke(() =>
                    {
                        model.ThumbnailBitmap = bitmap;
                    });
                });
            }

            this.SelectedProgramListModel.Items.Add(model);
        }

        #endregion

        #region SceneTemplateUpdateCommand -- 场景模板更新命令

        /// <summary>
        /// 场景模板更新命令
        /// </summary>
        public VCommand SceneTemplateUpdateCommand { get; set; }

        /// <summary>
        /// 场景模板更新
        /// </summary>
        private void SceneTemplateUpdate()
        {

        }

        #endregion

        #region SceneTemplateDeleteCommand -- 场景模板删除命令

        /// <summary>
        /// 场景模板删除命令
        /// </summary>
        public VCommand SceneTemplateDeleteCommand { get; set; }

        /// <summary>
        /// 场景模板删除
        /// </summary>
        private void SceneTemplateDelete()
        {
            if (this.SelectedSceneTemplateModel == null)
                return;

            this.SceneTemplateModels.Remove(this.SelectedSceneTemplateModel);
        }

        #endregion

        // ======================================================================================
        // Message
        // ======================================================================================

        /// <summary>
        /// 项目改变消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProjectChangedMessage(ProjectChangedMessage msg)
        {
            this.SceneTemplateModels = null;
            this.ProgramListModels = null;

            this.HasProjectDomain = msg.NewProjectDomain != null;
            if (msg.NewProjectDomain == null)
                return;

            this.IsProgramTemplateLoading = true;
            this.IsProgramListLoading = true;

            ThreadHelper.SafeRun(() =>
            {
                // 场景模板
                var sceneTemplateModels = this.programService.LoadProgramTemplateModels();

                // 其他模板
                var otherTemplateModels = new ObservableCollection<ProgramTemplateModel>();

                ProgramTemplateModel cmd = new ProgramTemplateModel(null);
                cmd.TemplateType = Storage.ProgramTemplateType.Command;
                cmd.Remark = "命令模板";
                cmd.EngineType = Storage.TVPEngineType.VIZ;
                cmd.ThumbnailBitmap = VIZ.TVP.Module.Resource.Properties.Resources.cmd_template;
                otherTemplateModels.Add(cmd);

                // 节目单
                var programListModels = this.programService.LoadProgramListModels();

                // 更新界面
                WPFHelper.BeginInvoke(() =>
                {
                    this.SceneTemplateModels = sceneTemplateModels;
                    this.OtherTemplateModels = otherTemplateModels;
                    this.ProgramListModels = programListModels;

                    this.IsProgramTemplateLoading = false;
                    this.IsProgramListLoading = false;
                });

                // 获取场景模板缩略图
                foreach (var model in sceneTemplateModels)
                {
                    var bitmap = ghService.GetImage(model.Thumbnail);

                    WPFHelper.BeginInvoke(() =>
                    {
                        model.ThumbnailBitmap = bitmap;
                    });
                }

                // 获取节目单项的缩略图
                foreach (var listModel in programListModels)
                {
                    foreach (var listItemModel in listModel.Items)
                    {
                        if (listItemModel.TemplateType == Storage.ProgramTemplateType.Scene)
                        {
                            var bitmap = ghService.GetImage(listItemModel.Thumbnail);
                            WPFHelper.BeginInvoke(() =>
                            {
                                listItemModel.ThumbnailBitmap = bitmap;
                            });
                        }
                        else if (listItemModel.TemplateType == Storage.ProgramTemplateType.Command)
                        {
                            WPFHelper.BeginInvoke(() =>
                            {
                                listItemModel.ThumbnailBitmap = Resource.Properties.Resources.cmd_template;
                            });
                        }
                    }
                }
            });
        }

        /// <summary>
        /// 项目保存消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnProjectSaveMessage(ProjectSaveMessage msg)
        {
            this.IsProgramTemplateLoading = true;
            this.IsProgramListLoading = true;

            ThreadHelper.SafeRun(() =>
            {
                if (this.SceneTemplateModels != null)
                {
                    this.programService.SaveProgramTemplateModels(this.SceneTemplateModels);
                }

                if (this.ProgramListModels != null)
                {
                    this.programService.SaveProgramListModels(this.ProgramListModels);
                }

                WPFHelper.BeginInvoke(() =>
                {
                    this.IsProgramTemplateLoading = false;
                    this.IsProgramListLoading = false;
                });
            });
        }

        // ======================================================================================
        // Public Function
        // ======================================================================================

        /// <summary>
        /// 销毁
        /// </summary>
        public override void Dispose()
        {

        }

        /// <summary>
        /// 添加场景模板
        /// </summary>
        /// <param name="fileModel">文件模型</param>
        public void AddSceneTemplate(GHResourceFileModel fileModel)
        {
            if (this.SceneTemplateModels.Any(p => p.SceneName == fileModel.Name))
                return;

            ProgramTemplateModel model = new ProgramTemplateModel(new Storage.ProgramTemplateEntity());
            model.TemplateID = Guid.NewGuid().ToString();
            model.SceneName = fileModel.Name;
            model.TemplateType = Storage.ProgramTemplateType.Scene;
            model.Thumbnail = fileModel.Thumbnail;
            model.ThumbnailBitmap = fileModel.ThumbnailBitmap;
            if (model.ThumbnailBitmap == null)
            {
                ThreadHelper.SafeRun(() =>
                {
                    var bitmap = this.ghService.GetImage(model.Thumbnail);

                    WPFHelper.BeginInvoke(() =>
                    {
                        model.ThumbnailBitmap = bitmap;
                    });
                });
            }

            model.EngineType = Storage.TVPEngineType.VIZ;

            this.SceneTemplateModels.Add(model);
        }
    }
}
