﻿using DevExpress.Mvvm.POCO;
using DevExpress.Mvvm.UI.Native.ViewGenerator;
using DevExpress.Xpf.Core.Native;
using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.Framework.Plugin;
using VIZ.TVP.Domain;
using VIZ.TVP.Service;
using VIZ.TVP.Storage;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// VIZ资源视图模型
    /// </summary>
    public class VizResourceViewModel : PluginViewModelBase, IVizResourceFileSupport
    {
        /// <summary>
        /// 日志
        /// </summary>
        private static readonly ILog log = LogManager.GetLogger(typeof(VizResourceViewModel));

        public VizResourceViewModel()
        {
            // 初始化控制器
            this.initController();

            // 初始化命令
            this.initCommand();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void initCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
            this.RefreshFolderCommand = new VCommand(this.RefreshFolder);
            this.RefreshFileCommand = new VCommand(this.RefreshFile);
            this.FolderExpandCommand = new VCommand<DevExpress.Xpf.Grid.TreeList.NodeDoubleClickEventArgs>(this.FolderExpand);

            this.FileContextMenuOpendCommand = new VCommand(this.FileContextMenuOpend);
            this.AddProgramTemplateCommand = new VCommand(this.AddProgramTemplate, this.CanAddProgramTemplate);
        }

        /// <summary>
        /// 初始化控制器
        /// </summary>
        private void initController()
        {
            this.vizResourceFileController = new VizResourceFileController(this);
        }

        // ==================================================================================
        // Property
        // ==================================================================================

        #region FolderModels -- 文件夹目录集合

        private ObservableCollection<GHResourceFolderModel> folderModels;
        /// <summary>
        /// 文件夹目录集合
        /// </summary>
        public ObservableCollection<GHResourceFolderModel> FolderModels
        {
            get { return folderModels; }
            set { folderModels = value; this.RaisePropertyChanged(nameof(FolderModels)); }
        }

        #endregion

        #region SelectedFolderModel -- 当前选中的文件夹

        private GHResourceFolderModel selectedFolderModel;
        /// <summary>
        /// 当前选中的文件夹
        /// </summary>
        public GHResourceFolderModel SelectedFolderModel
        {
            get { return selectedFolderModel; }
            set
            {
                selectedFolderModel = value;
                this.RaisePropertyChanged(nameof(SelectedFolderModel));
                // 更新文件模型
                this.vizResourceFileController.UpdateFileModels(value);
            }
        }

        #endregion

        #region FileModels -- 文件集合

        private ObservableCollection<GHResourceFileModel> fileModels;
        /// <summary>
        /// 文件集合
        /// </summary>
        public ObservableCollection<GHResourceFileModel> FileModels
        {
            get { return fileModels; }
            set { fileModels = value; this.RaisePropertyChanged(nameof(FileModels)); }
        }

        #endregion

        #region SelectedFileModel -- 选中的文件模型

        private GHResourceFileModel selectedFileModel;
        /// <summary>
        /// 选中的文件模型
        /// </summary>
        public GHResourceFileModel SelectedFileModel
        {
            get { return selectedFileModel; }
            set { selectedFileModel = value; this.RaisePropertyChanged(nameof(SelectedFileModel)); }
        }

        #endregion

        #region IsFolderLoading -- 是否正在加载文件夹

        private bool isFolderLoading;
        /// <summary>
        /// 是否正在加载文件夹
        /// </summary>
        public bool IsFolderLoading
        {
            get { return isFolderLoading; }
            set { isFolderLoading = value; this.RaisePropertyChanged(nameof(IsFolderLoading)); }
        }

        #endregion

        #region IsFileLoading -- 是否正在加载文件

        private bool isFileLoading;
        /// <summary>
        /// 是否正在加载文件
        /// </summary>
        public bool IsFileLoading
        {
            get { return isFileLoading; }
            set { isFileLoading = value; this.RaisePropertyChanged(nameof(IsFileLoading)); }
        }

        #endregion

        // ==================================================================================
        // Service & Controller
        // ==================================================================================

        /// <summary>
        /// GH 资源服务
        /// </summary>
        private IGHResourceService ghResourceService = new GHResourceService();

        /// <summary>
        /// VIZ资源文件控制器
        /// </summary>
        private VizResourceFileController vizResourceFileController;

        // ==================================================================================
        // Command
        // ==================================================================================

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {
            if (this.IsAlreadyLoaded)
                return;

            this.IsFolderLoading = true;

            // 刷新文件夹
            this.RefreshFolder();

            this.IsAlreadyLoaded = true;
        }

        #endregion

        #region RefreshFolderCommand -- 刷新文件夹命令

        /// <summary>
        /// 刷新文件夹命令
        /// </summary>
        public VCommand RefreshFolderCommand { get; set; }

        /// <summary>
        /// 刷新文件夹
        /// </summary>
        private void RefreshFolder()
        {
            this.IsFolderLoading = true;

            Task.Run(() =>
            {
                try
                {
                    List<GHResourceFolderModel> list = this.ghResourceService.GetGHResourceFolders("http://localhost:19398/folders/");

                    WPFHelper.BeginInvoke(() =>
                    {
                        this.FolderModels = list.FirstOrDefault()?.Children.Select(p => (GHResourceFolderModel)p).ToObservableCollection();

                        this.IsFolderLoading = false;
                    });
                }
                catch (Exception ex)
                {
                    log.Error(ex);
                }
            });
        }

        #endregion

        #region RefreshFileCommand -- 刷新文件命令

        /// <summary>
        /// 刷新文件命令
        /// </summary>
        public VCommand RefreshFileCommand { get; set; }

        /// <summary>
        /// 刷新文件
        /// </summary>
        private void RefreshFile()
        {
            GHResourceFolderModel folder = this.SelectedFolderModel;

            if (folder == null)
                return;

            this.vizResourceFileController.DisposeFileModels(folder);

            folder.IsRefreshedFiles = false;

            this.vizResourceFileController.UpdateFileModels(this.SelectedFolderModel);
        }

        #endregion

        #region FolderExpandCommand -- 文件夹展开命令

        /// <summary>
        /// 文件夹展开命令
        /// </summary>
        public VCommand<DevExpress.Xpf.Grid.TreeList.NodeDoubleClickEventArgs> FolderExpandCommand { get; set; }

        /// <summary>
        /// 文件夹展开
        /// </summary>
        private void FolderExpand(DevExpress.Xpf.Grid.TreeList.NodeDoubleClickEventArgs e)
        {
            if (this.SelectedFolderModel == null || e.ChangedButton != System.Windows.Input.MouseButton.Left)
                return;

            this.SelectedFolderModel.IsExpand = !this.SelectedFolderModel.IsExpand;
        }

        #endregion

        #region FileContextMenuOpendCommand -- 文件右键菜单打开命令

        /// <summary>
        /// 文件右键菜单打开命令
        /// </summary>
        public VCommand FileContextMenuOpendCommand { get; set; }

        /// <summary>
        /// 文件右键菜单打开
        /// </summary>
        private void FileContextMenuOpend()
        {
            this.AddProgramTemplateCommand.RaiseCanExecute();
        }

        #endregion

        #region AddProgramTemplateCommand -- 添加节目模板命令

        /// <summary>
        /// 添加节目模板命令
        /// </summary>
        public VCommand AddProgramTemplateCommand { get; set; }

        /// <summary>
        /// 是否可以执行添加节目模板
        /// </summary>
        /// <returns>是否可以执行添加节目模板</returns>
        private bool CanAddProgramTemplate()
        {
            return this.SelectedFileModel != null && this.SelectedFileModel.FileType == ResourceFileType.SCENE && ApplicationDomainEx.CurrentProjectDomain != null;
        }

        /// <summary>
        /// 添加节目模板
        /// </summary>
        private void AddProgramTemplate()
        {
            if (this.SelectedFileModel == null || this.SelectedFileModel.FileType != ResourceFileType.SCENE)
                return;

            IProgramListViewService service = ApplicationDomainEx.ServiceManager.GetService<IProgramListViewService>(ServiceKeys.PROGRAM_LIST_VIEW_SERVICE);
            if (service == null)
                return;

            service.AddSceneTemplate(this.SelectedFileModel);
        }

        #endregion

        // ==================================================================================
        // Public Function
        // ==================================================================================

        /// <summary>
        /// 销毁
        /// </summary>
        public override void Dispose()
        {

        }


        // ==================================================================================
        // Private Function
        // ==================================================================================


    }
}
