﻿using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.TVP.Domain;
using VIZ.TVP.Service;
using VIZ.TVP.Storage;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// 控制设置视图模型
    /// </summary>
    public class ControlSettingViewModel : ViewModelBase, ISettingViewModel
    {
        /// <summary>
        /// 日志
        /// </summary>
        private static readonly ILog log = LogManager.GetLogger(typeof(ControlSettingViewModel));

        public ControlSettingViewModel()
        {
            // 初始化命令
            this.initCommand();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void initCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
            this.AddGroupCommand = new VCommand(this.AddGroup);
            this.DeleteGroupCommand = new VCommand(this.DeleteGroup);
            this.EnabledGroupCommand = new VCommand(this.EnabledGroup);
            this.AddItemCommand = new VCommand(this.AddItem);
            this.DeleteItemCommand = new VCommand(this.DeleteItem);
        }

        // ======================================================================================
        // Field
        // ======================================================================================

        /// <summary>
        /// 包装连接服务
        /// </summary>
        private ITVPConnectionService tvpConnectionService = new TVPConnectionService();

        // ======================================================================================
        // Property
        // ======================================================================================

        #region Groups -- 分组集合

        private ObservableCollection<TVPConnectionGroupModel> groups;
        /// <summary>
        /// 分组集合
        /// </summary>
        public ObservableCollection<TVPConnectionGroupModel> Groups
        {
            get { return groups; }
            set { groups = value; this.RaisePropertyChanged(nameof(Groups)); }
        }

        #endregion

        #region SelectedGroup -- 当前选中的分组

        private TVPConnectionGroupModel selectedGroup;
        /// <summary>
        /// 当前选中的分组
        /// </summary>
        public TVPConnectionGroupModel SelectedGroup
        {
            get { return selectedGroup; }
            set { selectedGroup = value; this.RaisePropertyChanged(nameof(SelectedGroup)); }
        }

        #endregion

        #region SelectedItem -- 当前选中的项

        private TVPConnectionModel selectedItem;
        /// <summary>
        /// 当前选中的项
        /// </summary>
        public TVPConnectionModel SelectedItem
        {
            get { return selectedItem; }
            set { selectedItem = value; this.RaisePropertyChanged(nameof(SelectedItem)); }
        }

        #endregion

        // ======================================================================================
        // Command
        // ======================================================================================

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {
            if (this.IsAlreadyLoaded)
                return;

            this.IsAlreadyLoaded = true;

            this.Groups = this.tvpConnectionService.GetGroups();
        }

        #endregion

        #region AddGroupCommand -- 添加分组命令

        /// <summary>
        /// 添加分组命令
        /// </summary>
        public VCommand AddGroupCommand { get; set; }

        /// <summary>
        /// 添加分组
        /// </summary>
        private void AddGroup()
        {
            TVPConnectionGroupModel group = new TVPConnectionGroupModel(new Storage.TVPConnectionGroupEntity());
            group.GroupID = Guid.NewGuid();
            group.Name = "新建分组";
            this.Groups.Add(group);
        }

        #endregion

        #region DeleteGroupCommand -- 删除分组命令

        /// <summary>
        /// 删除分组 
        /// </summary>
        public VCommand DeleteGroupCommand { get; set; }

        /// <summary>
        /// 删除分组
        /// </summary>
        private void DeleteGroup()
        {
            if (this.SelectedGroup == null)
                return;

            this.Groups.Remove(this.SelectedGroup);
        }

        #endregion

        #region EnabledGroupCommand -- 启用组命令

        /// <summary>
        /// 启用组命令
        /// </summary>
        public VCommand EnabledGroupCommand { get; set; }

        /// <summary>
        /// 启用组
        /// </summary>
        private void EnabledGroup()
        {
            if (this.SelectedGroup == null)
                return;

            foreach (var group in this.Groups)
            {
                group.IsEnabled = group == this.SelectedGroup;
            }
        }

        #endregion

        #region AddItemCommand -- 添加项命令

        /// <summary>
        /// 添加项命令
        /// </summary>
        public VCommand AddItemCommand { get; set; }

        /// <summary>
        /// 添加项
        /// </summary>
        private void AddItem()
        {
            if (this.SelectedGroup == null)
                return;

            TVPConnectionModel item = new TVPConnectionModel(new Storage.TVPConnectionEntity());
            item.GroupID = this.SelectedGroup.GroupID;
            item.IP = "127.0.0.1";
            item.Port = 6100;

            this.SelectedGroup.Items.Add(item);
        }

        #endregion

        #region DeleteItemCommand -- 删除项命令

        /// <summary>
        /// 删除项命令
        /// </summary>
        public VCommand DeleteItemCommand { get; set; }

        /// <summary>
        /// 删除项
        /// </summary>
        private void DeleteItem()
        {
            if (this.SelectedItem == null || this.SelectedGroup == null)
                return;

            this.SelectedGroup.Items.Remove(this.SelectedItem);
        }

        #endregion

        // ======================================================================================
        // Public Function
        // ======================================================================================

        /// <summary>
        /// 保存
        /// </summary>
        public void Save()
        {
            this.tvpConnectionService.SaveGroups(this.Groups);
            ApplicationDomainEx.TVPConnectionManager.ConnectionGroups = this.Groups;
        }
    }
}
