﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Dynamic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using VIZ.Framework.Core;
using VIZ.TVP.Domain;
using VIZ.TVP.Service;
using VIZ.TVP.Storage;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// 控制对象控制器
    /// </summary>
    public class ControlObjectController
    {
        /// <summary>
        /// 控制对象控制器
        /// </summary>
        /// <param name="support">支持</param>
        public ControlObjectController(IControlObjectSupport support)
        {
            this.Support = support;
        }

        public IControlObjectSupport Support { get; private set; }

        /// <summary>
        /// ControlObject命令服务
        /// </summary>
        private IVizCommandControlObjectService vizCommandControlObjectService = new VizCommandControlObjectService();

        /// <summary>
        /// 从本地信息中更新控制对象字段
        /// </summary>
        /// <param name="programListItemModel">节目单项</param>
        /// <param name="fields">控制对象字段</param>
        public void UpdateControlObjectFieldFromLocal(ProgramListItemModel programListItemModel, IList<VizControlObjectFieldModel> fields)
        {
            if (programListItemModel == null || fields == null)
                return;

            foreach (VizControlObjectFieldEntity entity in programListItemModel.Entity.Fields)
            {
                VizControlObjectFieldModel field = fields.FirstOrDefault(p => p.TreeNodePath == entity.TreeNodePath && p.FieldIdentifier == entity.FieldIdentifier);
                if (field == null)
                    continue;

                field._value = entity.Value;
            }
        }

        /// <summary>
        /// 从VIZ中获取控制对象
        /// </summary>
        /// <param name="programListItemModel">节目单项模型</param>
        /// <param name="isUseLocalField">是否使用本地字段数据</param>
        public void GetControlObjectFromViz(ProgramListItemModel programListItemModel, bool isUseLocalField)
        {
            this.Support.IsLoading = true;

            // 清理属性改变事件
            this.CancelListValuePropertyChanged();

            // 否则加载控制对象列表
            ThreadHelper.SafeRun(() =>
            {
                // 从VIZ中获取控制对象列表
                var result = this.vizCommandControlObjectService.GetControlObjectList(ApplicationDomainEx.TVPConnectionManager.LocalConnection, programListItemModel, VizSceneLayerEnum.MAIN_SCENE);

                // 从本地数据更新控制对象字段
                if (isUseLocalField)
                {
                    this.UpdateControlObjectFieldFromLocal(programListItemModel, result.FirstOrDefault()?.Fields);
                }

                WPFHelper.BeginInvoke(() =>
                {
                    programListItemModel.ControlObjectList.Clear();

                    foreach (var item in result)
                    {
                        programListItemModel.ControlObjectList.Add(item);
                    }

                    if (ApplicationDomainEx.SelectionManager.ProgramListItemModel == programListItemModel)
                    {
                        this.Support.SelectedControlObject = programListItemModel.ControlObjectList.FirstOrDefault();
                    }

                    programListItemModel.IsLoadedControlObjectList = true;

                    this.Support.IsLoading = false;

                    // 推送属性值至
                    this.vizCommandControlObjectService.SetControlObjectValue(ApplicationDomainEx.TVPConnectionManager.LocalConnection, VizSceneLayerEnum.MAIN_SCENE, this.Support.SelectedControlObject);
                });
            });
        }

        /// <summary>
        /// 更新选中的控制对象字段
        /// </summary>
        /// <param name="view">视图</param>
        public void UpdateSelectedControlObjectField(FrameworkElement view)
        {
            VizControlObjectFieldModel field = this.Support.SelectedControlObjectField;
            if (field == null || field.Type != Storage.VizControlObjectFieldType.list)
            {
                this.Support.ListValueColumns = null;
                // 注销属性改变事件
                this.CancelListValuePropertyChanged();

                this.Support.ListValue = null;
                return;
            }

            if (!field.IsXmlValueLoaded)
            {
                this.CancelListValuePropertyChanged();
                List<GridColumnDefinition> columns;
                List<ExpandoObject> items;
                vizCommandControlObjectService.GetControlObjectXmlData(field.Value, field.FieldSchema.Schema, out columns, out items);
                field.ListValueColumns = columns;
                field.ListValue = items;
            }

            foreach (var node in field.FieldSchema.Schema.Fields)
            {
                GridColumnDefinition column = field.ListValueColumns.FirstOrDefault(p => p.FieldName == node.Name);
                if (column == null)
                    continue;

                VizControlObjectFieldType columnType = this.vizCommandControlObjectService.GetControlObjectFieldType(node.Type);

                switch (columnType)
                {
                    case VizControlObjectFieldType.none:
                        break;
                    case VizControlObjectFieldType.text:
                        column.CellEditTemplate = view.FindResource("ColumnTemplate_Text") as DataTemplate;
                        break;
                    case VizControlObjectFieldType.boolean:
                        column.CellEditTemplate = view.FindResource("ColumnTemplate_Boolean") as DataTemplate;
                        column.Width = new DevExpress.Xpf.Grid.GridColumnWidth(80);
                        break;
                    case VizControlObjectFieldType.richtext:
                        column.CellEditTemplate = view.FindResource("ColumnTemplate_Text") as DataTemplate;
                        break;
                    case VizControlObjectFieldType.image:
                        column.CellEditTemplate = view.FindResource("ColumnTemplate_Image") as DataTemplate;
                        break;
                    case VizControlObjectFieldType.list:
                        break;
                }
            }

            // 绑定属性
            foreach (INotifyPropertyChanged item in field.ListValue)
            {
                item.PropertyChanged -= this.Item_PropertyChanged;
                item.PropertyChanged += this.Item_PropertyChanged;
            }

            this.Support.ListValueColumns = field.ListValueColumns;
            this.Support.ListValue = field.ListValue;
        }

        /// <summary>
        /// 取消ListValue属性改变事件
        /// </summary>
        public void CancelListValuePropertyChanged()
        {
            // 注销属性改变事件
            if (this.Support.ListValue != null)
            {
                foreach (INotifyPropertyChanged item in this.Support.ListValue)
                {
                    item.PropertyChanged -= this.Item_PropertyChanged;
                }
            }
        }

        /// <summary>
        /// 是否是同一种类型
        /// </summary>
        /// <param name="fileType">文件类型</param>
        /// <param name="fieldType">字段类型</param>
        /// <returns>是否是同一种类型</returns>
        public bool IsSameType(ResourceFileType fileType, VizControlObjectFieldType fieldType)
        {
            if (fileType == ResourceFileType.IMAGE && fieldType == VizControlObjectFieldType.image)
                return true;

            return false;
        }

        /// <summary>
        /// 动态数据项属性改变时触发
        /// </summary>
        private void Item_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (this.Support.SelectedControlObjectField.Type != VizControlObjectFieldType.list)
                return;

            this.Support.SelectedControlObjectField.Value = this.vizCommandControlObjectService.GetControlObjectXml(this.Support.ListValue);
        }
    }
}
