﻿using log4net;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.TVP.Connection;
using VIZ.TVP.Domain;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// 主视图底部工具条视图模型
    /// </summary>
    public class MainViewBottomBarViewModel : ViewModelBase
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(MainViewBottomBarViewModel));

        public MainViewBottomBarViewModel()
        {
            // 初始化命令
            this.initCommand();

            // 初始化消息
            this.initMessage();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void initCommand()
        {
            this.LoadedCommand = new VCommand(this.Loaded);
            this.ItemClickCommand = new VCommand<TVPConnectionModel>(this.ItemClick);
        }

        /// <summary>
        /// 初始化消息
        /// </summary>
        private void initMessage()
        {
            ApplicationDomainEx.MessageManager.Register<TVPConnectionChangedMessage>(this, this.OnTVPConnectionChangedMessage);
            ApplicationDomainEx.MessageManager.Register<VizRenderReadyMessage>(this, this.OnVizRenderReadyMessage);
        }

        // ===================================================================
        // Property
        // ===================================================================

        #region LocalConnection -- 本机连接

        private TVPConnectionModel localConnection;
        /// <summary>
        /// 本机连接
        /// </summary>
        public TVPConnectionModel LocalConnection
        {
            get { return localConnection; }
            set { localConnection = value; this.RaisePropertyChanged(nameof(LocalConnection)); }
        }

        #endregion

        #region Items -- 连接项集合

        private ObservableCollection<TVPConnectionModel> items;
        /// <summary>
        /// 连接项集合
        /// </summary>
        public ObservableCollection<TVPConnectionModel> Items
        {
            get { return items; }
            set { items = value; this.RaisePropertyChanged(nameof(Items)); }
        }

        #endregion

        // ===================================================================
        // Command
        // ===================================================================

        #region LoadedCommand -- 加载命令

        /// <summary>
        /// 加载命令
        /// </summary>
        public VCommand LoadedCommand { get; set; }

        /// <summary>
        /// 加载
        /// </summary>
        private void Loaded()
        {
            if (this.IsAlreadyLoaded)
                return;

            this.IsAlreadyLoaded = true;

            TVPConnectionGroupModel group = ApplicationDomainEx.TVPConnectionManager.ConnectionGroups.FirstOrDefault(p => p.IsEnabled);
            if (group != null)
            {
                this.Items = group.Items.Where(p => p.IsEnabled).ToObservableCollection();
            }

            // 连接当前分组
            this.ConnectCurrentGroup();
        }

        #endregion

        #region ItemClickCommand -- 项点击命令

        /// <summary>
        /// 项点击命令
        /// </summary>
        public VCommand<TVPConnectionModel> ItemClickCommand { get; set; }

        /// <summary>
        /// 项点击命令
        /// </summary>
        /// <param name="model">连接模型</param>
        private void ItemClick(TVPConnectionModel model)
        {
            model.IsLoading = true;

            ThreadHelper.SafeRun(action: () =>
            {
                if (model.IsConnected)
                {
                    model.EndpointManager.Disconnect();
                }
                else
                {
                    model.EndpointManager.Connect();
                }
            }, final: () =>
            {
                model.IsLoading = false;
            });
        }

        #endregion

        // ===================================================================
        // Message
        // ===================================================================

        /// <summary>
        /// Viz渲染引擎准备完毕消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnVizRenderReadyMessage(VizRenderReadyMessage msg)
        {
            this.LocalConnection = ApplicationDomainEx.TVPConnectionManager.LocalConnection;
        }

        /// <summary>
        /// TVP连接改变消息
        /// </summary>
        /// <param name="msg">消息</param>
        private void OnTVPConnectionChangedMessage(TVPConnectionChangedMessage msg)
        {
            TVPConnectionGroupModel group = msg.ConnectionGroups.FirstOrDefault(p => p.IsEnabled);

            if (group != null)
            {
                this.Items = group.Items.Where(p => p.IsEnabled).ToObservableCollection();
            }
            else
            {
                this.Items = null;
            }

            // 连接当前分组
            this.ConnectCurrentGroup();
        }

        // ===================================================================
        // Public Function
        // ===================================================================

        /// <summary>
        /// 连接当前分组
        /// </summary>
        public void ConnectCurrentGroup()
        {
            if (ApplicationDomainEx.TVPConnectionManager.ConnectionGroups == null)
                return;

            var group = ApplicationDomainEx.TVPConnectionManager.ConnectionGroups.FirstOrDefault(p => p.IsEnabled);
            if (group == null)
                return;

            foreach (var item in group.Items)
            {
                if (item.IsConnected || !item.IsEnabled)
                    continue;

                ThreadHelper.SafeRun(action: () =>
                {
                    item.IsLoading = true;

                    if (item.EndpointManager == null)
                    {
                        item.InitEndpointManager(new VizEndpointManager(item.Remark, item.IP, item.Port));
                    }

                    item.EndpointManager.Connect();

                }, final: () =>
                {
                    item.IsLoading = false;
                });
            }
        }
    }
}
