﻿using DevExpress.XtraPrinting.Native.Extensions;
using log4net;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.Framework.Core;
using VIZ.TVP.Domain;
using VIZ.TVP.Service;
using VIZ.TVP.Storage;

namespace VIZ.TVP.Module
{
    /// <summary>
    /// VIZ资源文件控制器
    /// </summary>
    public class VizResourceFileController
    {
        /// <summary>
        /// 日志
        /// </summary>
        private static readonly ILog log = LogManager.GetLogger(typeof(VizResourceFileController));

        /// <summary>
        /// VIZ资源缩略图控制器
        /// </summary>
        /// <param name="support">支持</param>
        public VizResourceFileController(IVizResourceFileSupport support)
        {
            this.Support = support;
        }

        /// <summary>
        /// 支持
        /// </summary>
        public IVizResourceFileSupport Support { get; private set; }

        /// <summary>
        /// GH 资源服务
        /// </summary>
        private IGHResourceService ghResourceService = new GHResourceService();

        /// <summary>
        /// GH 服务
        /// </summary>
        private IGHService ghService = new GHService();

        /// <summary>
        /// 更新文件模型
        /// </summary>
        /// <param name="folder">文件夹</param>
        public void UpdateFileModels(GHResourceFolderModel folder)
        {
            // 文件夹对象不存在
            if (folder == null)
            {
                this.Support.FileModels = null;
                this.Support.SelectedFileModel = null;
                return;
            }

            // 已经获取过文件
            if (folder.IsRefreshedFiles)
            {
                this.Support.FileModels = folder.Files;
                return;
            }

            GH_Link_Node link_related = folder.EntryNode.links.FirstOrDefault(p => p.rel == GH_Link_Rel_Enums.related);
            if (link_related == null || string.IsNullOrWhiteSpace(link_related.href))
            {
                folder.IsRefreshedFiles = true;
                return;
            }

            this.Support.IsFileLoading = true;

            ThreadHelper.SafeRun(() =>
            {
                List<GHResourceFileModel> files = ghResourceService.GetGHResourceFiles(link_related.href, folder);

                WPFHelper.BeginInvoke(() =>
                {
                    folder.Files = files.ToObservableCollection();
                    this.Support.FileModels = folder.Files;
                    folder.IsRefreshedFiles = true;

                    this.Support.IsFileLoading = false;

                    // 开始下载缩略图
                    this.BeginDownloadThumbnail(folder);
                });
            });
        }

        /// <summary>
        /// 开始下载缩略图
        /// </summary>
        /// <param name="folder">文件夹</param>
        public void BeginDownloadThumbnail(GHResourceFolderModel folder)
        {
            if (folder == null || folder.Files == null || folder.Files.Count == 0)
                return;

            ThreadHelper.SafeRun(() =>
            {
                foreach (GHResourceFileModel file in folder.Files)
                {
                    this.DownloadThumbnail(file);
                }
            });
        }

        /// <summary>
        /// 开始下载缩略图
        /// </summary>
        /// <param name="file">文件</param>
        public void DownloadThumbnail(GHResourceFileModel file)
        {
            if (file == null || string.IsNullOrWhiteSpace(file.Thumbnail))
                return;

            try
            {
                Bitmap bmp = this.ghService.GetImage(file.Thumbnail);

                WPFHelper.BeginInvoke(() =>
                {
                    file.ThumbnailBitmap = bmp;
                });
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }
        }

        /// <summary>
        /// 销毁文件模型
        /// </summary>
        public void DisposeFileModels(GHResourceFolderModel folder)
        {
            if (folder == null || folder.Files == null || folder.Files.Count == 0)
                return;

            foreach (GHResourceFileModel file in folder.Files)
            {
                file.Dispose();
            }
        }
    }
}
