﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VIZ.TVP.Domain;
using VIZ.TVP.Storage;

namespace VIZ.TVP.Service
{
    /// <summary>
    /// VIZ命令服务
    /// </summary>
    public class VizCommandService : IVizCommandService
    {
        /// <summary>
        /// 设置场景
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="scene">场景</param>
        /// <param name="layer">图层</param>
        public void SetObject(TVPConnectionModel connection, string scene, VizLayerEnum layer)
        {
            if (connection == null)
                throw new ArgumentNullException(nameof(connection));

            connection.EndpointManager.Send($"RENDERER*{layer} SET_OBJECT SCENE*{scene}");
        }

        /// <summary>
        /// 播放
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="scene">场景</param>
        /// <param name="layer">图层</param>
        public void Start(TVPConnectionModel connection, string scene, VizLayerEnum layer)
        {
            if (connection == null)
                throw new ArgumentNullException(nameof(connection));

            if (string.IsNullOrWhiteSpace(scene))
            {
                connection.EndpointManager.Send($"RENDERER*{layer}*STAGE START");
            }
            else
            {
                connection.EndpointManager.Send($"RENDERER*{layer}*STAGE*DIRECTOR*{scene} START");
            }
        }

        /// <summary>
        /// 继续
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="scene">场景</param>
        /// <param name="layer">图层</param>
        public void Continue(TVPConnectionModel connection, string scene, VizLayerEnum layer)
        {
            if (connection == null)
                throw new ArgumentNullException(nameof(connection));

            if (string.IsNullOrWhiteSpace(scene))
            {
                connection.EndpointManager.Send($"RENDERER*{layer}*STAGE CONTINUE");
            }
            else
            {
                connection.EndpointManager.Send($"RENDERER*{layer}*STAGE*DIRECTOR*{scene} CONTINUE");
            }
        }

        /// <summary>
        /// 停止
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="scene">场景</param>
        /// <param name="layer">图层</param>
        public void Stop(TVPConnectionModel connection, string scene, VizLayerEnum layer)
        {
            if (connection == null)
                throw new ArgumentNullException(nameof(connection));

            if (string.IsNullOrWhiteSpace(scene))
            {
                connection.EndpointManager.Send($"RENDERER*{layer}*STAGE STOP");
            }
            else
            {
                connection.EndpointManager.Send($"RENDERER*{layer}*STAGE*DIRECTOR*{scene} STOP");
            }
        }

        /// <summary>
        /// 显示安全框
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="show">是否显示</param>
        public void ShowTS(TVPConnectionModel connection, bool show)
        {
            if (connection == null)
                throw new ArgumentNullException(nameof(connection));

            LocalInfoEntity info = ApplicationDomainEx.DataBaseManager.LocalInfo;

            if (info.LocalEngineType == LocalEngineType.Eng3)
            {
                connection.EndpointManager.Send($"RENDERER SET_TITLE_AREA {(show ? 1 : 0)}");
            }
            else if (info.LocalEngineType == LocalEngineType.Eng4)
            {
                connection.EndpointManager.Send($"{ApplicationDomainEx.LocalVizManager.VizRendererName} SET_TITLE_AREA {(show ? 1 : 0)}");
            }
        }

        /// <summary>
        /// 显示安区域
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="show">是否显示</param>
        public void ShowSA(TVPConnectionModel connection, bool show)
        {
            if (connection == null)
                throw new ArgumentNullException(nameof(connection));

            LocalInfoEntity info = ApplicationDomainEx.DataBaseManager.LocalInfo;

            if (info.LocalEngineType == LocalEngineType.Eng3)
            {
                connection.EndpointManager.Send($"RENDERER SET_SAFE_AREA {(show ? 1 : 0)}");
            }
            else if (info.LocalEngineType == LocalEngineType.Eng4)
            {
                connection.EndpointManager.Send($"{ApplicationDomainEx.LocalVizManager.VizRendererName} SET_SAFE_AREA {(show ? 1 : 0)}");
            }
        }

        /// <summary>
        /// 显示边界框
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="show">是否显示</param>
        public void ShowBB(TVPConnectionModel connection, bool show)
        {
            if (connection == null)
                throw new ArgumentNullException(nameof(connection));

            LocalInfoEntity info = ApplicationDomainEx.DataBaseManager.LocalInfo;

            if (info.LocalEngineType == LocalEngineType.Eng3)
            {
                connection.EndpointManager.Send($"RENDERER SHOW_BOUNDING_BOX {(show ? 1 : 0)}");
            }
            else if (info.LocalEngineType == LocalEngineType.Eng4)
            {
                connection.EndpointManager.Send($"{ApplicationDomainEx.LocalVizManager.VizRendererName} SHOW_BOUNDING_BOX {(show ? 1 : 0)}");
            }
        }

        /// <summary>
        /// 显示RGB
        /// </summary>
        /// <param name="connection">连接</param>
        public void ShowRGB(TVPConnectionModel connection)
        {
            if (connection == null)
                throw new ArgumentNullException(nameof(connection));

            LocalInfoEntity info = ApplicationDomainEx.DataBaseManager.LocalInfo;

            if (info.LocalEngineType == LocalEngineType.Eng3)
            {
                connection.EndpointManager.Send($"RENDERER SET_KEY 0,RENDERER*KEY_INTERNAL*ACTIVE SET 0");
            }
            else if (info.LocalEngineType == LocalEngineType.Eng4)
            {
                connection.EndpointManager.Send($"{ApplicationDomainEx.LocalVizManager.VizRendererName} SET_KEY 0,RENDERER*KEY_INTERNAL*ACTIVE SET 0");
            }
        }

        /// <summary>
        /// 显示键
        /// </summary>
        /// <param name="connection">连接</param>
        public void ShowKey(TVPConnectionModel connection)
        {
            if (connection == null)
                throw new ArgumentNullException(nameof(connection));

            LocalInfoEntity info = ApplicationDomainEx.DataBaseManager.LocalInfo;

            if (info.LocalEngineType == LocalEngineType.Eng3)
            {
                connection.EndpointManager.Send($"RENDERER SET_KEY 1,RENDERER*KEY_INTERNAL*ACTIVE SET 0");
            }
            else if (info.LocalEngineType == LocalEngineType.Eng4)
            {
                connection.EndpointManager.Send($"{ApplicationDomainEx.LocalVizManager.VizRendererName} SET_KEY 1,RENDERER*KEY_INTERNAL*ACTIVE SET 0");
            }
        }

        /// <summary>
        /// 显示预览键
        /// </summary>
        /// <param name="connection">连接</param>
        public void ShowKeyPreview(TVPConnectionModel connection)
        {
            if (connection == null)
                throw new ArgumentNullException(nameof(connection));

            LocalInfoEntity info = ApplicationDomainEx.DataBaseManager.LocalInfo;

            if (info.LocalEngineType == LocalEngineType.Eng3)
            {
                connection.EndpointManager.Send($"RENDERER SET_KEY 0,RENDERER*KEY_INTERNAL*IMAGE RESET,RENDERER*KEY_INTERNAL*ACTIVE SET 1");
            }
            else if (info.LocalEngineType == LocalEngineType.Eng4)
            {
                connection.EndpointManager.Send($"{ApplicationDomainEx.LocalVizManager.VizRendererName} SET_KEY 0,RENDERER*KEY_INTERNAL*IMAGE RESET,RENDERER*KEY_INTERNAL*ACTIVE SET 1");
            }
        }

        /// <summary>
        /// 上版子
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="scene">场景</param>
        /// <param name="layer">层</param>s
        /// <param name="use_all_directors">使用所有控制器</param>
        public void TakeIn(TVPConnectionModel connection, string scene, VizLayerEnum layer, bool use_all_directors)
        {
            connection.EndpointManager.Send("RENDERER*UPDATE SET 0");
            connection.EndpointManager.Send($"RENDERER*{layer} SET_OBJECT {scene}");
            connection.EndpointManager.Send("RENDERER*UPDATE SET 1");

            if (use_all_directors)
            {
                connection.EndpointManager.Send($"RENDERER*{layer}*STAGE START");
            }
            else
            {
                connection.EndpointManager.Send($"RENDERER*{layer}*STAGE START");
            }
        }

        /// <summary>
        /// 继续版子
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="layer">层</param>s
        public void TakeContinue(TVPConnectionModel connection, VizLayerEnum layer)
        {
            connection.EndpointManager.Send($"RENDERER*UPDATE SET 1");
            connection.EndpointManager.Send($"RENDERER*{layer}*STAGE CONTINUE");
        }

        /// <summary>
        /// 下版子
        /// </summary>
        /// <param name="connection">连接</param>
        /// <param name="layer">层</param>s
        public void TakeOut(TVPConnectionModel connection, VizLayerEnum layer)
        {
            connection.EndpointManager.Send($"RENDERER*UPDATE SET 1");
            connection.EndpointManager.Send($"RENDERER*{layer} SET_OBJECT");
        }
    }
}
