﻿using log4net;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using VIZ.Framework.Core;
using VIZ.Framework.Storage;
using VIZ.TimeSlice.Domain;
using VIZ.TimeSlice.Storage;

namespace VIZ.TimeSlice.Module
{
    /// <summary>
    /// 主视图模型
    /// </summary>
    public class MainViewModel : ViewModelBase
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(MainViewModel));

        public MainViewModel()
        {
            // 初始化命令
            this.InitCommand();

            // 初始化属性
            this.InitProperty();
        }

        /// <summary>
        /// 初始化命令
        /// </summary>
        private void InitCommand()
        {
            this.StartCommand = new VCommand(this.Start);
            this.SelectWorkPathCommand = new VCommand(this.SelectWorkPath);
            this.SelectBackgroundImagePathCommand = new VCommand(this.SelectBackgroundImagePath);
            this.SelectDestPathCommand = new VCommand(this.SelectDestPath);
            this.RefreshCommand = new VCommand(this.Refresh);
        }

        /// <summary>
        /// 初始化属性
        /// </summary>
        private void InitProperty()
        {
            // --------------------------------------------
            // 数据集合
            // --------------------------------------------
            // 纯色集合
            this.SliceBackgroundItems = ApplicationDomainEx.CsvContext.SliceBackgrounds;
            // 分辨率集合
            this.SliceResolutionItems = ApplicationDomainEx.CsvContext.SliceResolutions;
            // 输出格式集合
            this.SliceFormatItems = ApplicationDomainEx.CsvContext.SliceFormats;
            // 背景颜色策略集合
            this.VideoBackgroundTypeItems = EnumHelper.GetEnumModels<SliceVideoBackgroundTypeEnum>();

            // --------------------------------------------
            // 数据项
            // --------------------------------------------
            // 工作路径
            this.workPath = ApplicationDomainEx.IniStorage.GetValue<SliceConfig, string>(p => p.path);
            // 背景方案
            int videobackground = ApplicationDomainEx.IniStorage.GetValue<SliceConfig, int>(p => p.videobackground);
            this.selectedVideoBackgroundType = this.VideoBackgroundTypeItems.FirstOrDefault(p => (int)(SliceVideoBackgroundTypeEnum)p.Key == videobackground);
            // 背景图片
            this.backgroundImagePath = ApplicationDomainEx.IniStorage.GetValue<SliceConfig, string>(p => p.imagepath);
            // 纯色背景
            string background = ApplicationDomainEx.IniStorage.GetValue<SliceConfig, string>(p => p.background);
            this.selectedSliceBackground = this.SliceBackgroundItems.FirstOrDefault(p => p.Name == background);
            // 绘制坐标
            this.positionX = ApplicationDomainEx.IniStorage.GetValue<SliceConfig, int>(p => p.positionX);
            this.positionY = ApplicationDomainEx.IniStorage.GetValue<SliceConfig, int>(p => p.positionY);
            // 每X帧保留1帧
            this.frames = ApplicationDomainEx.IniStorage.GetValue<SliceConfig, int>(p => p.frames);
            // 分辨率
            string resolution = ApplicationDomainEx.IniStorage.GetValue<SliceConfig, string>(p => p.resolution);
            this.selectedSliceResolution = this.SliceResolutionItems.FirstOrDefault(p => p.Name == resolution);
            // 帧率
            this.fps = ApplicationDomainEx.IniStorage.GetValue<SliceConfig, int>(p => p.fps);
            // 输出格式
            string format = ApplicationDomainEx.IniStorage.GetValue<SliceConfig, string>(p => p.format);
            this.selectedSliceFormat = this.SliceFormatItems.FirstOrDefault(p => p.Name == format);
        }

        // ==========================================================
        // Property
        // ==========================================================

        #region WorkPath -- 工作路径

        private string workPath;
        /// <summary>
        /// 工作路径
        /// </summary>
        public string WorkPath
        {
            get { return workPath; }
            set
            {
                workPath = value;
                this.RaisePropertyChanged(nameof(WorkPath));
                ApplicationDomainEx.IniStorage.SetValue<SliceConfig>(p => p.path, value);
            }
        }

        #endregion

        #region DestPath -- 目标路径

        private string destPath;
        /// <summary>
        /// 目标路径
        /// </summary>
        public string DestPath
        {
            get { return destPath; }
            set { destPath = value; this.RaisePropertyChanged(nameof(DestPath)); }
        }

        #endregion

        #region VideoBackgroundTypeItems -- 视频背景方案项集合

        private List<EnumModel> videoBackgroundTypeItems;
        /// <summary>
        /// 视频背景方案项集合
        /// <see cref="SliceVideoBackgroundTypeEnum"/>
        /// </summary>
        public List<EnumModel> VideoBackgroundTypeItems
        {
            get { return videoBackgroundTypeItems; }
            set { videoBackgroundTypeItems = value; this.RaisePropertyChanged(nameof(VideoBackgroundTypeItems)); }
        }

        #endregion

        #region SelectedVideoBackgroundType -- 选中的背景方案

        private EnumModel selectedVideoBackgroundType;
        /// <summary>
        /// 选中的背景方案
        /// <see cref="SliceVideoBackgroundTypeEnum"/>
        /// </summary>
        public EnumModel SelectedVideoBackgroundType
        {
            get { return selectedVideoBackgroundType; }
            set
            {
                selectedVideoBackgroundType = value;
                this.RaisePropertyChanged(nameof(SelectedVideoBackgroundType));
                ApplicationDomainEx.IniStorage.SetValue<SliceConfig>(p => p.videobackground, value == null ? 0 : (int)value.Key);
            }
        }

        #endregion

        #region BackgroundImagePath -- 背景图片路径

        private string backgroundImagePath;
        /// <summary>
        /// 背景图片路径
        /// </summary>
        public string BackgroundImagePath
        {
            get { return backgroundImagePath; }
            set
            {
                backgroundImagePath = value;
                this.RaisePropertyChanged(nameof(BackgroundImagePath));
                ApplicationDomainEx.IniStorage.SetValue<SliceConfig>(p => p.imagepath, value);
            }
        }

        #endregion

        #region SliceBackgroundItems -- 纯色集合

        private List<SliceBackground> sliceBackgroundItems;
        /// <summary>
        /// 纯色集合
        /// </summary>
        public List<SliceBackground> SliceBackgroundItems
        {
            get { return sliceBackgroundItems; }
            set { sliceBackgroundItems = value; this.RaisePropertyChanged(nameof(SliceBackgroundItems)); }
        }

        #endregion

        #region SelectedSliceBackground -- 选中的纯色

        private SliceBackground selectedSliceBackground;
        /// <summary>
        /// 选中的纯色
        /// </summary>
        public SliceBackground SelectedSliceBackground
        {
            get { return selectedSliceBackground; }
            set
            {
                selectedSliceBackground = value;
                this.RaisePropertyChanged(nameof(SelectedSliceBackground));
                ApplicationDomainEx.IniStorage.SetValue<SliceConfig>(p => p.background, value == null ? string.Empty : value.Name);
            }
        }

        #endregion

        #region PositionX -- 绘制坐标X值

        private int positionX;
        /// <summary>
        /// 绘制坐标X值
        /// </summary>
        public int PositionX
        {
            get { return positionX; }
            set
            {
                positionX = value;
                this.RaisePropertyChanged(nameof(PositionX));
                ApplicationDomainEx.IniStorage.SetValue<SliceConfig>(p => p.positionX, value);
            }
        }

        #endregion

        #region PositionY -- 绘制坐标Y值

        private int positionY;
        /// <summary>
        /// 绘制坐标Y值
        /// </summary>
        public int PositionY
        {
            get { return positionY; }
            set
            {
                positionY = value;
                this.RaisePropertyChanged(nameof(PositionY));
                ApplicationDomainEx.IniStorage.SetValue<SliceConfig>(p => p.positionY, value);
            }
        }

        #endregion

        #region Frames -- 每X帧保留1帧

        private int frames;
        /// <summary>
        /// 每X帧保留1帧
        /// </summary>
        public int Frames
        {
            get { return frames; }
            set
            {
                frames = value;
                this.RaisePropertyChanged(nameof(Frames));
                ApplicationDomainEx.IniStorage.SetValue<SliceConfig>(p => p.frames, value);
            }
        }

        #endregion

        #region SliceResolutionItems -- 分辨率集合

        private List<SliceResolution> sliceResolutionItems;
        /// <summary>
        /// 分辨率集合
        /// </summary>
        public List<SliceResolution> SliceResolutionItems
        {
            get { return sliceResolutionItems; }
            set { sliceResolutionItems = value; this.RaisePropertyChanged(nameof(SliceResolutionItems)); }
        }

        #endregion

        #region SelectedSliceResolution -- 选中的分辨率

        private SliceResolution selectedSliceResolution;
        /// <summary>
        /// 选中的分辨率
        /// </summary>
        public SliceResolution SelectedSliceResolution
        {
            get { return selectedSliceResolution; }
            set
            {
                selectedSliceResolution = value;
                this.RaisePropertyChanged(nameof(SelectedSliceResolution));
                ApplicationDomainEx.IniStorage.SetValue<SliceConfig>(p => p.resolution, value == null ? string.Empty : value.Name);
            }
        }

        #endregion

        #region Fps -- 帧率

        private int fps;
        /// <summary>
        /// 帧率
        /// </summary>
        public int Fps
        {
            get { return fps; }
            set
            {
                fps = value;
                this.RaisePropertyChanged(nameof(Fps));
                ApplicationDomainEx.IniStorage.SetValue<SliceConfig>(p => p.fps, value);
            }
        }

        #endregion

        #region SliceFormatItems -- 输出格式集合

        private List<SliceFormat> sliceFormatItems;
        /// <summary>
        /// 输出格式集合
        /// </summary>
        public List<SliceFormat> SliceFormatItems
        {
            get { return sliceFormatItems; }
            set { sliceFormatItems = value; this.RaisePropertyChanged(nameof(SliceFormatItems)); }
        }

        #endregion

        #region SelectedSliceFormat -- 选中的输出格式

        private SliceFormat selectedSliceFormat;
        /// <summary>
        /// 选中的输出格式
        /// </summary>
        public SliceFormat SelectedSliceFormat
        {
            get { return selectedSliceFormat; }
            set
            {
                selectedSliceFormat = value;
                this.RaisePropertyChanged(nameof(SelectedSliceFormat));
                ApplicationDomainEx.IniStorage.SetValue<SliceConfig>(p => p.format, value == null ? string.Empty : value.Name);
            }
        }

        #endregion

        #region LogString -- 日志字符串

        private string logString;
        /// <summary>
        /// 日志字符串
        /// </summary>
        public string LogString
        {
            get { return logString; }
            set { logString = value; this.RaisePropertySaveChanged(nameof(LogString)); }
        }

        #endregion

        #region IsStartButtonEnabled -- 开始按钮是否可用

        private bool isStartButtonEnabled = true;
        /// <summary>
        /// 开始按钮是否可用
        /// </summary>
        public bool IsStartButtonEnabled
        {
            get { return isStartButtonEnabled; }
            set { isStartButtonEnabled = value; this.RaisePropertySaveChanged(nameof(IsStartButtonEnabled)); }
        }

        #endregion

        // ==========================================================
        // Field
        // ==========================================================

        /// <summary>
        /// 推送消息地址
        /// </summary>
        private readonly static string CLIENT_PULL_MESSAGE_URL = ApplicationDomainEx.IniStorage.GetValue<ClientConfig, string>(p => p.CLIENT_PULL_MESSAGE_URL);

        /// <summary>
        /// 刷新配置地址
        /// </summary>
        private readonly static string CLIENT_REFRESH_INI_CONFIG_URL = ApplicationDomainEx.IniStorage.GetValue<ClientConfig, string>(p => p.CLIENT_REFRESH_INI_CONFIG_URL);

        // ==========================================================
        // Commond
        // ==========================================================

        #region SelectWorkPathCommand -- 选择工作路径命令

        /// <summary>
        /// 选择工作路径命令
        /// </summary>
        public VCommand SelectWorkPathCommand { get; set; }

        /// <summary>
        /// 选择工作路径
        /// </summary>
        private void SelectWorkPath()
        {
            FolderBrowserDialog fbd = new FolderBrowserDialog();
            if (fbd.ShowDialog() != DialogResult.OK)
                return;

            this.WorkPath = fbd.SelectedPath;
        }

        #endregion

        #region SelectBackgroundImagePathCommand -- 选择背景图片命令

        /// <summary>
        /// 选择背景图片命令
        /// </summary>
        public VCommand SelectBackgroundImagePathCommand { get; set; }

        /// <summary>
        /// 选择背景图片
        /// </summary>
        private void SelectBackgroundImagePath()
        {
            OpenFileDialog ofd = new OpenFileDialog();
            ofd.Filter = "图片文件(*.jpg,*.gif,*.bmp)|*.jpg;*.gif;*.bmp|所有文件|*.*";
            ofd.Multiselect = false;
            if (ofd.ShowDialog() != DialogResult.OK)
                return;

            this.BackgroundImagePath = ofd.FileName;
        }

        #endregion

        #region SelectDestPathCommand -- 选择目标目录命令

        /// <summary>
        /// 选择目标目录命令
        /// </summary>
        public VCommand SelectDestPathCommand { get; set; }

        /// <summary>
        /// 选择目标目录
        /// </summary>
        private void SelectDestPath()
        {
            FolderBrowserDialog fbd = new FolderBrowserDialog();
            fbd.SelectedPath = this.WorkPath;
            if (fbd.ShowDialog() != DialogResult.OK)
                return;

            string destPath = fbd.SelectedPath.Replace(this.WorkPath, string.Empty);
            if (destPath.StartsWith("\\"))
            {
                destPath = destPath.Substring(1);
            }
            this.DestPath = destPath;
        }

        #endregion

        #region StartCommand -- 开始命令

        /// <summary>
        /// 开始命令
        /// </summary>
        public VCommand StartCommand { get; set; }

        /// <summary>
        /// 开始
        /// </summary>
        private void Start()
        {
            this.IsStartButtonEnabled = false;

            Task.Run(() =>
            {
                try
                {
                    string url = CLIENT_PULL_MESSAGE_URL.Replace("{path}", this.DestPath);

                    this.LogString = $"{DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss")} -- HTTP请求：{url}";

                    string result = HttpHelper.Get(url, null);

                    this.LogString = $"{this.LogString}\r\n{DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss")} -- 返回: {result}";
                }
                catch (Exception ex)
                {
                    log.Error(ex);

                    this.LogString = $"{this.LogString}\r\n{DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss")} -- 异常: {ex.Message}";
                }
                finally
                {
                    this.IsStartButtonEnabled = true;
                }
            });
        }

        #endregion

        #region RefreshCommand -- 刷新配置命令

        /// <summary>
        /// 刷新配置命令
        /// </summary>
        public VCommand RefreshCommand { get; set; }

        /// <summary>
        /// 刷新配置
        /// </summary>
        private void Refresh()
        {
            this.IsStartButtonEnabled = false;

            Task.Run(() =>
            {
                try
                {
                    string url = CLIENT_REFRESH_INI_CONFIG_URL;

                    this.LogString = $"{DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss")} -- HTTP请求：{url}";

                    string result = HttpHelper.Get(url, null);

                    this.LogString = $"{this.LogString}\r\n{DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss")} -- 返回: {result}";
                }
                catch (Exception ex)
                {
                    log.Error(ex);

                    this.LogString = $"{this.LogString}\r\n{DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss")} -- 异常: {ex.Message}";
                }
                finally
                {
                    this.IsStartButtonEnabled = true;
                }
            });
        }

        #endregion

        // ==========================================================
        // Private Function
        // ==========================================================

    }
}
