﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Input;
using log4net;

namespace VIZ.Framework.Core
{
    /// <summary>
    /// 命令
    /// </summary>
    public class VCommand : ICommand, IDisposable
    {
        /// <summary>
        /// 日志
        /// </summary>
        private static ILog log = LogManager.GetLogger(typeof(VCommand));

        /// <summary>
        /// 命令
        /// </summary>
        /// <param name="execute">执行命令</param>
        public VCommand(Action execute)
        {
            this._Execute = execute;
        }

        /// <summary>
        /// 命令
        /// </summary>
        /// <param name="execute">执行命令</param>
        /// <param name="canExecute">是否可以执行命令</param>
        public VCommand(Action execute, Func<bool> canExecute)
        {
            this._Execute = execute;
            this._CanExecute = canExecute;
        }

        /// <summary>
        /// 是否可以执行改变时触发
        /// </summary>
        public event EventHandler CanExecuteChanged
        {
            add
            {
                if (this._CanExecute != null)
                {
                    CommandManager.RequerySuggested += value;
                }
            }

            remove
            {
                if (this._CanExecute != null)
                {
                    CommandManager.RequerySuggested -= value;
                }
            }
        }

        /// <summary>
        /// 通知界面是否可以执行命令
        /// </summary>
        public void RaiseCanExecute()
        {
            CommandManager.InvalidateRequerySuggested();
        }

        /// <summary>
        /// 是否可以执行命令
        /// </summary>
        /// <param name="parameter">参数</param>
        /// <returns>是否可以执行</returns>
        public bool CanExecute(object parameter)
        {
            try
            {
                if (this._CanExecute == null)
                {
                    return true;
                }

                bool result = this._CanExecute();

                return result;
            }
            catch (Exception ex)
            {
                log.Error(ex);

                return false;
            }
        }

        /// <summary>
        /// 执行命令
        /// </summary>
        /// <param name="parameter">参数</param>
        public void Execute(object parameter)
        {
            try
            {
                if (this._Execute == null)
                    return;

                this._Execute();
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }
        }

        /// <summary>
        /// 销毁
        /// </summary>
        public void Dispose()
        {
            this._CanExecute = null;
            this._Execute = null;
        }

        /// <summary>
        /// 行为
        /// </summary>
        private Action _Execute;

        /// <summary>
        /// 是否可以执行行为
        /// </summary>
        private Func<bool> _CanExecute;
    }

    /// <summary>
    /// 命令
    /// </summary>
    /// <typeparam name="T">命令参数类型</typeparam>
    public class VCommand<T> : ICommand, IDisposable
    {
        /// <summary>
        /// 日志
        /// </summary>
        private static ILog log = LogManager.GetLogger(typeof(VCommand<T>));

        /// <summary>
        /// 命令
        /// </summary>
        /// <param name="execute">执行命令</param>
        public VCommand(Action<T> execute)
        {
            this._Execute = execute;
        }

        /// <summary>
        /// 命令
        /// </summary>
        /// <param name="execute">执行命令</param>
        /// <param name="canExecute">是否可以执行命令</param>
        public VCommand(Action<T> execute, Func<T, bool> canExecute)
        {
            this._Execute = execute;
            this._CanExecute = canExecute;
        }

        /// <summary>
        /// 是否可以执行改变时触发
        /// </summary>
        public event EventHandler CanExecuteChanged;

        /// <summary>
        /// 通知界面是否可以执行命令
        /// </summary>
        public void RaiseCanExecute()
        {
            if (this.CanExecuteChanged != null)
            {
                this.CanExecuteChanged(this, new EventArgs());
            }
        }

        /// <summary>
        /// 是否可以执行命令
        /// </summary>
        /// <param name="parameter">参数</param>
        /// <returns>是否可以执行</returns>
        public bool CanExecute(object parameter)
        {
            try
            {
                if (this._CanExecute == null)
                {
                    return true;
                }

                bool result = this._CanExecute((T)parameter);

                return result;
            }
            catch (Exception ex)
            {
                log.Error(ex);

                return false;
            }
        }

        /// <summary>
        /// 执行命令
        /// </summary>
        /// <param name="parameter">参数</param>
        public void Execute(object parameter)
        {
            try
            {
                if (this._Execute == null)
                    return;

                this._Execute((T)parameter);
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }
        }

        /// <summary>
        /// 销毁
        /// </summary>
        public void Dispose()
        {
            this._CanExecute = null;
            this._Execute = null;
        }

        /// <summary>
        /// 行为
        /// </summary>
        private Action<T> _Execute;

        /// <summary>
        /// 是否可以执行行为
        /// </summary>
        private Func<T, bool> _CanExecute;
    }
}
