﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using log4net;

namespace VIZ.Framework.Common
{
    /// <summary>
    /// 板卡流
    /// </summary>
    public class BMDStream : VideoStreamBase<BMDStreamOption>
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(BMDStream));

        /// <summary>
        /// NDI流
        /// </summary>
        /// <param name="option">设置</param>
        public BMDStream(BMDStreamOption option)
            : base(option)
        {
            // 处理视频帧任务
            this.TaskDic.Add(BMDStreamTaskNames.EXECUTE_VIDEO, new BMDStreamExecuteVideoTask(this));
        }

        /* ========================================================================================================= */
        /* === Const === */
        /* ========================================================================================================= */

        /// <summary>
        /// 视频接收回调
        /// </summary>
        /// <param name="name">名称</param>
        /// <param name="data">视频帧数据 BGRA 8bit</param>
        /// <param name="width">视频宽度</param>
        /// <param name="height">视频高度</param>
        public delegate void SendVideoCallBack(string name, IntPtr data, int width, int height);

        /// <summary>
        /// 测试回调
        /// </summary>
        /// <param name="name">名称</param>
        public delegate void TestCallBack(string name);

        /// <summary>
        /// 初始化板卡
        /// </summary>
        /// <param name="callback">接收视频回调</param>
        /// <returns></returns>
        [DllImport("libSDIPackage.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern bool InitBMD(SendVideoCallBack callback);

        /// <summary>
        /// 测试
        /// </summary>
        /// <param name="callback">回调</param>
        [DllImport("libSDIPackage.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern void Test(TestCallBack callback);

        /* ========================================================================================================= */
        /* === Property === */
        /* ========================================================================================================= */


        /* ========================================================================================================= */
        /* === Internal Field === */
        /* ========================================================================================================= */


        /* ========================================================================================================= */
        /* === Field === */
        /* ========================================================================================================= */

        /// <summary>
        /// 任务池
        /// </summary>
        private Dictionary<BMDStreamTaskNames, BMDStreamTaskBase> TaskDic = new Dictionary<BMDStreamTaskNames, BMDStreamTaskBase>();

        /* ========================================================================================================= */
        /* === Function === */
        /* ========================================================================================================= */

        /// <summary>
        /// 开始
        /// </summary>
        public void Start()
        {
            // 启动NDI视频帧处理任务
            this.TaskDic[BMDStreamTaskNames.EXECUTE_VIDEO].Start();

            // 初始化板卡
            InitBMD(new SendVideoCallBack(this.ExecuteSendVideoCallBack));
        }

        /// <summary>
        /// 停止
        /// </summary>
        public void Stop()
        {
            // 停止所有任务
            foreach (var task in this.TaskDic.Values)
            {
                task.Stop();
            }

            // TODO: 停止板卡
        }

        /// <summary>
        /// 销毁
        /// </summary>
        public override void Dispose()
        {
            this.Stop();
        }

        /// <summary>
        /// 执行视频帧接收回调函数
        /// </summary>
        /// <param name="name">名称</param>
        /// <param name="data">视频帧数据 BGRA 8bit</param>
        /// <param name="width">视频宽度</param>
        /// <param name="height">视频高度</param>
        private void ExecuteSendVideoCallBack(string name, IntPtr data, int width, int height)
        {
            try
            {
                BMDStreamVideoFrame videoFrame = new BMDStreamVideoFrame();
                videoFrame.Width = width;
                videoFrame.Height = height;
                videoFrame.Length = width * height * 4;
                //videoFrame.TimeStamp = 
                videoFrame.DataStream = new SharpDX.DataStream(videoFrame.Length, true, true);
                unsafe
                {
                    Buffer.MemoryCopy(data.ToPointer(), videoFrame.DataStream.DataPointer.ToPointer(), videoFrame.Length, videoFrame.Length);
                }

                this.VideoFrameQueue.Enqueue(videoFrame);

                Thread.Sleep(10);
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }
        }
    }
}