﻿/* -LICENSE-START-
 ** Copyright (c) 2018 Blackmagic Design
 **
 ** Permission is hereby granted, free of charge, to any person or organization
 ** obtaining a copy of the software and accompanying documentation covered by
 ** this license (the "Software") to use, reproduce, display, distribute,
 ** execute, and transmit the Software, and to prepare derivative works of the
 ** Software, and to permit third-parties to whom the Software is furnished to
 ** do so, all subject to the following:
 **
 ** The copyright notices in the Software and this entire statement, including
 ** the above license grant, this restriction and the following disclaimer,
 ** must be included in all copies of the Software, in whole or in part, and
 ** all derivative works of the Software, unless such copies or derivative
 ** works are solely in the form of machine-executable object code generated by
 ** a source language processor.
 **
 ** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 ** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 ** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
 ** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
 ** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
 ** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 ** DEALINGS IN THE SOFTWARE.
 ** -LICENSE-END-
 */
using System;
using System.Diagnostics;
using DeckLinkAPI;

namespace VIZ.Framework.Common
{
    #region eventargs
    public class DeckLinkInputFormatChangedEventArgs : EventArgs
    {
        public readonly _BMDVideoInputFormatChangedEvents notificationEvents;
        public readonly _BMDDisplayMode displayMode;
        public readonly _BMDPixelFormat pixelFormat;

        public DeckLinkInputFormatChangedEventArgs(_BMDVideoInputFormatChangedEvents notificationEvents, _BMDDisplayMode displayMode, _BMDPixelFormat pixelFormat)
        {
            this.notificationEvents = notificationEvents;
            this.displayMode = displayMode;
            this.pixelFormat = pixelFormat;
        }
    }

    /// <summary>
    /// DeckLink视频帧接收事件参数
    /// </summary>
    public class DeckLinkVideoFrameArrivedEventArgs : EventArgs
    {
        /// <summary>
        /// 视频帧
        /// </summary>
        public readonly IDeckLinkVideoInputFrame videoFrame;

        /// <summary>
        /// 输入是否有效
        /// </summary>
        public readonly bool inputInvalid;

        /// <summary>
        /// 帧时间
        /// </summary>
        public readonly long frameTime;

        /// <summary>
        /// DeckLink视频帧接收事件参数
        /// </summary>
        /// <param name="videoFrame">视频帧</param>
        /// <param name="inputInvalid">输入是否有效</param>
        /// <param name="frameTime">帧时间</param>
        public DeckLinkVideoFrameArrivedEventArgs(IDeckLinkVideoInputFrame videoFrame, bool inputInvalid, long frameTime)
        {
            this.videoFrame = videoFrame;
            this.inputInvalid = inputInvalid;
            this.frameTime = frameTime;
        }
    }
    #endregion

    /// <summary>
    /// DeckLink 设备
    /// </summary>
    public class DeckLinkInputDevice : IDeckLinkInputCallback
    {
        /// <summary>
        /// DeckLink
        /// </summary>
        public IDeckLink DeckLink { get; private set; }

        /// <summary>
        /// DeckLink 输入
        /// </summary>
        public IDeckLinkInput DeckLinkInput { get; private set; }

        /// <summary>
        /// DeckLink
        /// </summary>
        public IDeckLinkProfileAttributes DeckLinkProfileAttributes { get; private set; }

        /// <summary>
        /// 显示名称
        /// </summary>
        public string DisplayName { get; private set; }

        /// <summary>
        /// 显示模式
        /// </summary>
        public _BMDDisplayMode DisplayMode { get; private set; }

        /// <summary>
        /// 视频帧格式
        /// </summary>
        public _BMDPixelFormat PixelFormat { get; private set; }

        /// <summary>
        /// 之前输入信号是否缺失
        /// </summary>
        private bool m_prevInputSignalAbsent = false;

        /// <summary>
        /// 时间刻度
        /// </summary>
        private long timeScale = 1001;

        /// <summary>
        /// DeckLink输入设备
        /// </summary>
        /// <param name="deckLink">DeckLink</param>
        public DeckLinkInputDevice(IDeckLink deckLink)
        {
            this.DeckLink = deckLink;
            // Query input interface
            this.DeckLinkInput = deckLink as IDeckLinkInput;
            this.DeckLinkProfileAttributes = deckLink as IDeckLinkProfileAttributes;

            deckLink.GetDisplayName(out string displayName);
            this.DisplayName = displayName;
        }

        public event EventHandler<DeckLinkInputFormatChangedEventArgs> InputFormatChangedHandler;
        public event EventHandler<DeckLinkVideoFrameArrivedEventArgs> VideoFrameArrivedHandler;

        /// <summary>
        /// 开始捕获
        /// </summary>
        public void StartCapture()
        {
            var videoInputFlags = _BMDVideoInputFlags.bmdVideoInputFlagDefault | _BMDVideoInputFlags.bmdVideoInputEnableFormatDetection;

            this.DisplayMode = _BMDDisplayMode.bmdModeNTSC;
            this.PixelFormat = _BMDPixelFormat.bmdFormat8BitYUV;

            // Set capture callback
            this.DeckLinkInput.SetCallback(this);

            // Set the video input mode
            this.DeckLinkInput.EnableVideoInput(this.DisplayMode, this.PixelFormat, videoInputFlags);

            // Start the capture
            this.DeckLinkInput.StartStreams();
        }

        /// <summary>
        /// 输入格式改变时触发
        /// </summary>
        /// <param name="notificationEvents">输入格式改变事件</param>
        /// <param name="newDisplayMode">新的显示模式</param>
        /// <param name="detectedSignalFlags">输入格式标志</param>
        void IDeckLinkInputCallback.VideoInputFormatChanged(_BMDVideoInputFormatChangedEvents notificationEvents, IDeckLinkDisplayMode newDisplayMode, _BMDDetectedVideoInputFormatFlags detectedSignalFlags)
        {
            // Restart capture with the new video mode if told to
            var pixelFormat = _BMDPixelFormat.bmdFormat8BitYUV;
            if (detectedSignalFlags.HasFlag(_BMDDetectedVideoInputFormatFlags.bmdDetectedVideoInputRGB444))
                pixelFormat = _BMDPixelFormat.bmdFormat8BitBGRA;

            _BMDDisplayMode displayMode = newDisplayMode.GetDisplayMode();

            if (this.DisplayMode == displayMode && this.PixelFormat == PixelFormat)
                return;

            this.DisplayMode = displayMode;
            this.PixelFormat = pixelFormat;

            // Stop the capture
            this.DeckLinkInput.StopStreams();

            // Set the video input mode
            this.DeckLinkInput.EnableVideoInput(displayMode, pixelFormat, _BMDVideoInputFlags.bmdVideoInputEnableFormatDetection);

            // Start the capture
            this.DeckLinkInput.StartStreams();

            string displayModeStr;
            newDisplayMode.GetName(out displayModeStr);

            long timeValue;
            newDisplayMode.GetFrameRate(out timeValue, out timeScale);

            // Debug.WriteLine("Video input display mode changed to " + displayModeStr);

            // Register input format changed event
            var handler = InputFormatChangedHandler;
            if (handler != null)
            {
                handler.Invoke(this, new DeckLinkInputFormatChangedEventArgs(notificationEvents, displayMode, pixelFormat));
            }
        }

        /// <summary>
        /// 捕获帧数据时触发
        /// </summary>
        /// <param name="videoFrame">视频帧</param>
        /// <param name="audioPacket">音频数据包</param>
        void IDeckLinkInputCallback.VideoInputFrameArrived(IDeckLinkVideoInputFrame videoFrame, IDeckLinkAudioInputPacket audioPacket)
        {
            if (videoFrame != null)
            {
                bool inputSignalAbsent = videoFrame.GetFlags().HasFlag(_BMDFrameFlags.bmdFrameHasNoInputSource);

                // Detect change in input signal, restart stream when valid stream detected 
                bool restartStream = !inputSignalAbsent && m_prevInputSignalAbsent;
                if (restartStream)
                {
                    this.DeckLinkInput.StopStreams();
                    this.DeckLinkInput.FlushStreams();
                    this.DeckLinkInput.StartStreams();
                }

                m_prevInputSignalAbsent = inputSignalAbsent;

                var frameWidth = videoFrame.GetWidth();
                var frameHeight = videoFrame.GetHeight();

                long frameTime, frameDuration;
                videoFrame.GetStreamTime(out frameTime, out frameDuration, timeScale);

                //Console.WriteLine(value: "Frame received #" + frameCount++ + "; Frame time = " + frameTime +
                //    "; Frame size = " + frameWidth + "x" + frameHeight + "; " + (inputSignalAbsent ? "Invalid" : "Valid")
                //    + (restartStream ? " - restarting" : ""));

                // Register video frame received event
                if (this.VideoFrameArrivedHandler != null)
                {
                    this.VideoFrameArrivedHandler.Invoke(this, new DeckLinkVideoFrameArrivedEventArgs(videoFrame, inputSignalAbsent, frameTime));
                }
            }

            System.Runtime.InteropServices.Marshal.ReleaseComObject(videoFrame);
        }

        /// <summary>
        /// 停止捕获
        /// </summary>
        public void StopCapture()
        {
            // Remove all listenders
            InputFormatChangedHandler = null;
            VideoFrameArrivedHandler = null;

            // Stop the capture
            this.DeckLinkInput.StopStreams();

            // Disable video input
            this.DeckLinkInput.DisableVideoInput();

            // Disable callbacks
            this.DeckLinkInput.SetCallback(null);
        }
    }
}
