﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using DeckLinkAPI;
using log4net;
using VIZ.Framework.Core;

namespace VIZ.Framework.Common
{
    /// <summary>
    /// DeckLink 管理器任务项
    /// </summary>
    public class DeckLinkManagerTaskItem
    {
        /// <summary>
        /// 行为
        /// </summary>
        public Action Action { get; set; }

        /// <summary>
        /// 异常
        /// </summary>
        public Exception Exception { get; set; }

        /// <summary>
        /// 是否处理完成
        /// </summary>
        public bool IsFinished { get; set; }

        /// <summary>
        /// 重置事件
        /// </summary>
        public ManualResetEvent ResetEvent { get; set; }

        /// <summary>
        /// 超时时间
        /// </summary>
        public TimeSpan Timeout { get; set; } = TimeSpan.FromSeconds(5);
    }

    /// <summary>
    /// DeckLink 管理器
    /// </summary>
    public static class DeckLinkManager
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(DeckLinkManager));

        /// <summary>
        /// 设备发现
        /// </summary>
        public static DeckLinkDeviceNotification DeviceNotification { get; private set; }

        /// <summary>
        /// 设备信息集合
        /// </summary>
        public static List<DeckLinkInputDevice> DeckLinks { get; private set; } = new List<DeckLinkInputDevice>();

        /// <summary>
        /// DeckLink主线程
        /// </summary>
        private static Thread DeckLinkMainThread;

        /// <summary>
        /// DeckLink主线程信息
        /// </summary>
        private static TaskInfo DeckLinkMainThreadInfo;

        /// <summary>
        /// 任务队列
        /// </summary>
        private static ConcurrentQueue<DeckLinkManagerTaskItem> TaskQueue = new ConcurrentQueue<DeckLinkManagerTaskItem>();

        /// <summary>
        /// 开始执行
        /// </summary>
        /// <param name="action">行为</param>
        public static void BeginInvoke(Action action)
        {
            DeckLinkManagerTaskItem task = new DeckLinkManagerTaskItem();
            task.Action = action;
            TaskQueue.Enqueue(task);
        }

        /// <summary>
        /// 执行
        /// </summary>
        /// <param name="action">行为</param>
        public static void Invoke(Action action)
        {
            DeckLinkManagerTaskItem task = new DeckLinkManagerTaskItem();
            task.Action = action;
            task.ResetEvent = new ManualResetEvent(false);
            TaskQueue.Enqueue(task);
            task.ResetEvent.WaitOne(task.Timeout);
        }

        /// <summary>
        /// 初始化
        /// </summary>
        public static void Init()
        {
            DeckLinkMainThreadInfo = new TaskInfo();
            DeckLinkMainThread = new Thread(ExecuteDeckLinkMainThread);
            DeckLinkMainThread.SetApartmentState(ApartmentState.MTA);
            DeckLinkMainThread.Start(DeckLinkMainThreadInfo);

            Invoke(() =>
            {
                DeviceNotification = new DeckLinkDeviceNotification();
                DeviceNotification.deviceArrived += DeviceNotification_deviceArrived;
                DeviceNotification.deviceRemoved += DeviceNotification_deviceRemoved;
            });
        }

        /// <summary>
        /// 销毁
        /// </summary>
        public static void Dispose()
        {
            if (DeviceNotification != null)
            {
                DeviceNotification.deviceArrived -= DeviceNotification_deviceArrived;
                DeviceNotification.deviceRemoved -= DeviceNotification_deviceRemoved;
                DeviceNotification = null;
            }

            if (DeckLinkMainThreadInfo != null)
            {
                DeckLinkMainThreadInfo.IsCancel = true;
                DeckLinkMainThreadInfo = null;
            }
        }

        /// <summary>
        /// 执行DeckLink主线程
        /// </summary>
        /// <param name="obj">参数</param>
        [MTAThread]
        private static void ExecuteDeckLinkMainThread(object obj)
        {
            try
            {
                TaskInfo info = obj as TaskInfo;

                while (!info.IsCancel)
                {
                    while (TaskQueue.Count > 0)
                    {
                        if (!TaskQueue.TryDequeue(out DeckLinkManagerTaskItem task))
                        {
                            Thread.Sleep(100);
                            continue;
                        }

                        try
                        {
                            task.Action();
                            task.ResetEvent?.Set();
                        }
                        catch (Exception ex)
                        {
                            log.Error(ex);
                        }
                        finally
                        {
                            task.IsFinished = true;
                        }
                    }

                    Thread.Sleep(100);
                }
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }
        }

        /// <summary>
        /// 设备发现
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private static void DeviceNotification_deviceArrived(object sender, DeckLinkDiscoveryEventArgs e)
        {
            lock (DeckLinks)
            {
                DeckLinks.Add(new DeckLinkInputDevice(e.deckLink));
            }
        }

        /// <summary>
        /// 设备移除
        /// </summary>
        private static void DeviceNotification_deviceRemoved(object sender, DeckLinkDiscoveryEventArgs e)
        {
            lock (DeckLinks)
            {
                DeckLinkInputDevice device = DeckLinks.FirstOrDefault(p => p.DeckLinkInput == e.deckLink);

                if (device != null)
                {
                    DeckLinks.Remove(device);
                }
            }
        }
    }
}
