﻿using log4net;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using VIZ.Framework.Core;
using VIZ.Framework.Domain;

namespace VIZ.Framework.Common
{
    /// <summary>
    /// 闪烁面板
    /// </summary>
    public class FlashingBorder : Border, IDisposable
    {
        /// <summary>
        /// 日志
        /// </summary>
        private readonly static ILog log = LogManager.GetLogger(typeof(FlashingBorder));

        /// <summary>
        /// 时间控件
        /// </summary>
        public FlashingBorder()
        {
            ApplicationDomain.ObjectPoolManager.Add($"FlashingBorder__{Guid.NewGuid()}", this);

            this.Loaded += FlashingBorder_Loaded;
        }

        #region IsFlashingEnabled -- 是否启用闪烁

        /// <summary>
        /// 是否启用闪烁
        /// </summary>
        public bool IsFlashingEnabled
        {
            get { return (bool)GetValue(IsFlashingEnabledProperty); }
            set { SetValue(IsFlashingEnabledProperty, value); }
        }

        /// <summary>
        /// Using a DependencyProperty as the backing store for IsFlashingEnabled.  This enables animation, styling, binding, etc...
        /// </summary>
        public static readonly DependencyProperty IsFlashingEnabledProperty =
            DependencyProperty.Register("IsFlashingEnabled", typeof(bool), typeof(FlashingBorder), new PropertyMetadata(false));

        #endregion

        #region FlashingInterval -- 闪烁间隔（单位：秒）

        /// <summary>
        /// 闪烁间隔（单位：秒）
        /// </summary>
        public TimeSpan FlashingInterval
        {
            get { return (TimeSpan)GetValue(FlashingIntervalProperty); }
            set { SetValue(FlashingIntervalProperty, value); }
        }

        /// <summary>
        /// Using a DependencyProperty as the backing store for FlashingInterval.  This enables animation, styling, binding, etc...
        /// </summary>
        public static readonly DependencyProperty FlashingIntervalProperty =
            DependencyProperty.Register("FlashingInterval", typeof(TimeSpan), typeof(FlashingBorder), new PropertyMetadata(TimeSpan.Zero));

        #endregion

        /// <summary>
        /// 更新任务信息
        /// </summary>
        private TaskInfo updateTaskInfo;

        /// <summary>
        /// 更新任务
        /// </summary>
        private Task updateTask;

        /// <summary>
        /// 销毁
        /// </summary>
        public void Dispose()
        {
            if (this.updateTaskInfo == null)
                return;

            this.updateTaskInfo.IsCancel = true;
            this.updateTaskInfo = null;
            this.updateTask = null;
        }

        /// <summary>
        /// 控件完成加载
        /// </summary>
        private void FlashingBorder_Loaded(object sender, System.Windows.RoutedEventArgs e)
        {
            if (this.updateTask != null || WPFHelper.IsInDesignMode(this))
                return;

            this.updateTaskInfo = new TaskInfo();
            this.updateTask = Task.Run(this.update);
        }

        /// <summary>
        /// 更新
        /// </summary>
        [DebuggerNonUserCode]
        private void update()
        {
            TaskInfo info = this.updateTaskInfo;
            if (info == null)
                return;

            TimeSpan interval = TimeSpan.FromSeconds(1);

            while (!info.IsCancel)
            {
                WPFHelper.Invoke(() =>
                {
                    if (!this.IsFlashingEnabled)
                    {
                        this.Visibility = Visibility.Visible;
                        return;
                    }

                    this.Visibility = this.Visibility == Visibility.Visible ? Visibility.Hidden : Visibility.Visible;

                    interval = this.FlashingInterval;
                });

                if (interval > TimeSpan.Zero)
                {
                    Task.Delay(interval).Wait();
                }
                else
                {
                    Task.Delay(1000).Wait();
                }
            }
        }
    }
}
