﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using VIZ.Framework.Core;

namespace VIZ.Framework.Common
{
    /// <summary>
    /// 消息框
    /// </summary>
    [TemplatePart(Name = nameof(PART_YesButton), Type = typeof(Button))]
    [TemplatePart(Name = nameof(PART_CancelButton), Type = typeof(Button))]
    public class MessageBoxEx : ContentControl
    {
        static MessageBoxEx()
        {
            DefaultStyleKeyProperty.OverrideMetadata(typeof(MessageBoxEx), new FrameworkPropertyMetadata(typeof(MessageBoxEx)));
        }

        #region PART

        /// <summary>
        /// 确定按钮
        /// </summary>
        private Button PART_YesButton;

        /// <summary>
        /// 取消按钮
        /// </summary>
        private Button PART_CancelButton;

        #endregion

        #region Buttons -- 按钮组合

        /// <summary>
        /// 按钮组合
        /// </summary>
        public MessageBoxExButtons Buttons
        {
            get { return (MessageBoxExButtons)GetValue(ButtonsProperty); }
            set { SetValue(ButtonsProperty, value); }
        }

        /// <summary>
        /// Using a DependencyProperty as the backing store for Buttons.  This enables animation, styling, binding, etc...
        /// </summary>
        public static readonly DependencyProperty ButtonsProperty =
            DependencyProperty.Register("Buttons", typeof(MessageBoxExButtons), typeof(MessageBoxEx), new PropertyMetadata(MessageBoxExButtons.YES));

        #endregion

        #region Title -- 标题

        /// <summary>
        /// 标题
        /// </summary>
        public string Title
        {
            get { return (string)GetValue(TitleProperty); }
            set { SetValue(TitleProperty, value); }
        }

        /// <summary>
        /// Using a DependencyProperty as the backing store for Title.  This enables animation, styling, binding, etc...
        /// </summary>
        public static readonly DependencyProperty TitleProperty =
            DependencyProperty.Register("Title", typeof(string), typeof(MessageBoxEx), new PropertyMetadata(null));

        #endregion

        #region Message -- 消息

        /// <summary>
        /// 消息
        /// </summary>
        public string Message
        {
            get { return (string)GetValue(MessageProperty); }
            set { SetValue(MessageProperty, value); }
        }

        /// <summary>
        /// Using a DependencyProperty as the backing store for Message.  This enables animation, styling, binding, etc...
        /// </summary>
        public static readonly DependencyProperty MessageProperty =
            DependencyProperty.Register("Message", typeof(string), typeof(MessageBoxEx), new PropertyMetadata(null));

        #endregion

        #region Result -- 返回值

        /// <summary>
        /// 返回值
        /// </summary>
        public MessageBoxExResult Result
        {
            get { return (MessageBoxExResult)GetValue(ResultProperty); }
            set { SetValue(ResultProperty, value); }
        }

        /// <summary>
        /// Using a DependencyProperty as the backing store for Result.  This enables animation, styling, binding, etc...
        /// </summary>
        public static readonly DependencyProperty ResultProperty =
            DependencyProperty.Register("Result", typeof(MessageBoxExResult), typeof(MessageBoxEx), new PropertyMetadata(MessageBoxExResult.YES));

        #endregion

        /// <summary>
        /// 创建模板时触发
        /// </summary>
        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            this.PART_YesButton = this.Template.FindName(nameof(PART_YesButton), this) as Button;
            this.PART_CancelButton = this.Template.FindName(nameof(PART_CancelButton), this) as Button;

            if (this.PART_YesButton != null)
            {
                this.PART_YesButton.Click -= PART_YesButton_Click;
                this.PART_YesButton.Click += PART_YesButton_Click;
            }

            if (this.PART_CancelButton != null)
            {
                this.PART_CancelButton.Click -= PART_CancelButton_Click;
                this.PART_CancelButton.Click += PART_CancelButton_Click;
            }
        }

        /// <summary>
        /// 点击取消按钮
        /// </summary>
        private void PART_CancelButton_Click(object sender, RoutedEventArgs e)
        {
            this.Result = MessageBoxExResult.CANCEL;
            Window window = WPFHelper.GetAncestorByType<Window>(this);
            window?.Close();
        }

        /// <summary>
        /// 点击确定按钮
        /// </summary>
        private void PART_YesButton_Click(object sender, RoutedEventArgs e)
        {
            this.Result = MessageBoxExResult.YES;
            Window window = WPFHelper.GetAncestorByType<Window>(this);
            window?.Close();
        }

        /// <summary>
        /// 显示提示框（线程安全）
        /// </summary>
        /// <param name="title">标题</param>
        /// <param name="message">消息</param>
        /// <param name="buttons">按钮组</param>
        /// <param name="ownerWindow">所属窗口</param>
        /// <returns>返回值</returns>
        public static MessageBoxExResult ShowDialog(string title, string message, MessageBoxExButtons buttons, Window ownerWindow)
        {
            MessageBoxExResult result = MessageBoxExResult.YES;

            WPFHelper.Invoke(() =>
            {
                MessageBoxExWindow window = new MessageBoxExWindow();
                window.box.Title = title;
                window.box.Message = message;
                window.box.Buttons = buttons;
                window.Owner = ownerWindow;

                window.ShowDialog();

                result = window.box.Result;
            });

            return result;
        }

        /// <summary>
        /// 显示提示框（线程安全）
        /// </summary>
        /// <param name="title">标题</param>
        /// <param name="message">消息</param>
        /// <param name="buttons">按钮组</param>
        /// <returns>返回值</returns>
        public static MessageBoxExResult ShowDialog(string title, string message, MessageBoxExButtons buttons)
        {
            MessageBoxExResult result = MessageBoxExResult.YES;

            WPFHelper.Invoke(() =>
            {
                MessageBoxExWindow window = new MessageBoxExWindow();
                window.box.Title = title;
                window.box.Message = message;
                window.box.Buttons = buttons;
                window.Owner = Application.Current?.MainWindow;

                window.ShowDialog();

                result = window.box.Result;
            });

            return result;
        }

        /// <summary>
        /// 显示提示框（线程安全）
        /// </summary>
        /// <param name="title">标题</param>
        /// <param name="message">消息</param>
        /// <returns>返回值</returns>
        public static MessageBoxExResult ShowDialog(string title, string message)
        {
            return ShowDialog(title, message, MessageBoxExButtons.YES);
        }

        /// <summary>
        /// 显示提示框（线程安全）
        /// </summary>
        /// <param name="message">消息</param>
        /// <returns>返回值</returns>
        public static MessageBoxExResult ShowDialog(string message)
        {
            return ShowDialog("提示", message);
        }

        /// <summary>
        /// 显示提示框（线程安全）
        /// </summary>
        /// <param name="message">消息</param>
        /// <param name="ownerWindow">所属窗口</param>
        /// <returns>返回值</returns>
        public static MessageBoxExResult ShowDialog(string message, Window ownerWindow)
        {
            return ShowDialog("提示", message, MessageBoxExButtons.YES, ownerWindow);
        }
    }
}
