﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using log4net;
using VIZ.Framework.Domain;

namespace VIZ.Framework.Connection
{
    /// <summary>
    /// 固定长度Buffer包解析器
    /// </summary>
    [ConnPackageProvider(ConnPackageProviderType.TCP | ConnPackageProviderType.UDP)]
    public class ConnFixedBufferPackageProvider : IConnPackageProvider
    {
        /// <summary>
        /// 日志
        /// </summary>
        private static ILog log = LogManager.GetLogger(typeof(ConnFixedBufferPackageProvider));

        /// <summary>
        /// 固定长度Buffer包解析器
        /// </summary>
        /// <param name="fixedBufferSize">固定Buffer长度</param>
        public ConnFixedBufferPackageProvider(int fixedBufferSize)
        {
            this.FixedBufferSize = fixedBufferSize;
            this.buffer_index = 0;
            this.buffer_cache = new byte[fixedBufferSize];
        }

        /// <summary>
        /// 固定Buffer长度
        /// </summary>
        public int FixedBufferSize { get; private set; }

        /// <summary>
        /// 数据缓存
        /// </summary>
        private byte[] buffer_cache;

        /// <summary>
        /// 数据缓存索引
        /// </summary>
        private int buffer_index;

        /// <summary>
        /// 执行
        /// </summary>
        /// <param name="package">数据包</param>
        public void Execute(ConnPackageInfo package)
        {
            int copy = Math.Min(package.DataSize, (buffer_cache.Length - buffer_index));
            Array.Copy(package.Data, 0, this.buffer_cache, buffer_index, copy);
            this.buffer_index += copy;
            if (this.buffer_index < this.FixedBufferSize)
                return;

            // 处理器处理消息
            ConnFixedBufferInfo info = new ConnFixedBufferInfo();
            info.LocalIP = package.LocalIP;
            info.LocalPort = package.LocalPort;
            info.RemoteIP = package.RemoteIP;
            info.RemotePort = package.RemotePort;
            info.Buffer = this.buffer_cache;

            this.buffer_index = 0;
            this.buffer_cache = new byte[this.FixedBufferSize];

            try
            {
                this.Execute(info);
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }

            // 发送消息
            ConnFixedBufferMessage msg = new ConnFixedBufferMessage();
            msg.LocalIP = package.LocalIP;
            msg.LocalPort = package.LocalPort;
            msg.RemoteIP = package.RemoteIP;
            msg.RemotePort = package.RemotePort;
            msg.Buffer = info.Buffer;

            try
            {
                ApplicationDomain.MessageManager.Send(msg);
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }
        }

        /// <summary>
        /// 执行
        /// </summary>
        /// <param name="info">信息</param>
        protected virtual void Execute(ConnFixedBufferInfo info)
        {
            // noting to do.
        }
    }
}